package code

import (
	"context"
	"fmt"

	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/chunker"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/types"

	codechunker "gitlab.com/gitlab-org/rust/gitlab-code-parser/bindings/go/chunker"
)

// Chunk information related to the file
type ChunkFileInfo struct {
	Name string
	OID  string
}

// Chunker implements the chunker.Chunker interface using code-parser chunking
type Chunker struct {
	options chunker.ChunkOptions
}

// New creates a new code chunker with the provided options
func New(options chunker.ChunkOptions) (chunker.Chunker, error) {
	chunkSize := options.ChunkSize
	if chunkSize <= 0 {
		chunkSize = chunker.DefaultChunkSize
	}

	// If chunk overlap is greater than chunk size, return an error
	if options.ChunkOverlap >= chunkSize {
		return nil, fmt.Errorf("chunk overlap (%d) must be less than chunk size (%d)", options.ChunkOverlap, chunkSize)
	}

	return &Chunker{
		options: chunker.ChunkOptions{
			ChunkSize:    chunkSize,
			ChunkOverlap: options.ChunkOverlap,
		},
	}, nil
}

// ChunkFiles implements the chunker.Chunker interface
func (c *Chunker) ChunkFiles(ctx context.Context, files []types.File) ([]chunker.Chunk, error) {
	// Initialize the split-code chunker
	// This chunker falls back to a size chunker if the language is not supported
	codeChunker, err := codechunker.NewChunkerSplitCode(int(c.options.ChunkSize))
	if err != nil {
		return nil, fmt.Errorf("codechunker.NewChunkerSplitCode: %w", err)
	}

	fileInfoMap := make(map[string]ChunkFileInfo)
	for _, file := range files {
		fileInfoMap[file.Path] = ChunkFileInfo{
			Name: file.GetFilename(),
			OID:  file.OID,
		}
		codeChunker.AddFile(file.Path, file.Content)
	}

	err = codeChunker.ChunkFiles()
	if err != nil {
		return nil, fmt.Errorf("codeChunker.ChunkFiles: %w", err)
	}

	chunks := make([]chunker.Chunk, 0)
	for codeChunk := range codeChunker.Chunks() {
		chunks = append(chunks, chunker.Chunk{
			Path:      codeChunk.FilePath,
			Type:      chunker.FileContentType,
			Name:      fileInfoMap[codeChunk.FilePath].Name,
			Language:  codeChunk.Language,
			Content:   codeChunk.Content,
			OID:       fileInfoMap[codeChunk.FilePath].OID,
			StartByte: codeChunk.StartByte,
			Length:    codeChunk.Length,
			StartLine: codeChunk.StartLine,
		})
	}

	codeChunker.Close()

	return chunks, nil
}
