package chunk

import (
	"context"
	"encoding/json"
	"fmt"
	"os"
	"strings"
	"testing"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/chunker"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/streamer"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/types"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/shared/gitaly"
)

func TestCreateChunker(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name        string
		options     Options
		expectError bool
	}{
		{
			name:        "valid options",
			options:     Options{ChunkSize: 1000, ChunkOverlap: 100},
			expectError: false,
		},
		{
			name:        "zero chunk size uses default",
			options:     Options{ChunkSize: 0, ChunkOverlap: 100},
			expectError: false,
		},
		{
			name:        "overlap equal to size should fail",
			options:     Options{ChunkSize: 100, ChunkOverlap: 100},
			expectError: true,
		},
		{
			name:        "overlap greater than size should fail",
			options:     Options{ChunkSize: 100, ChunkOverlap: 200},
			expectError: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			testChunker, err := createChunker(tt.options)

			if tt.expectError {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
				require.NotNil(t, testChunker)
			}
		})
	}
}

func TestCreateVectorStoreIndexer(t *testing.T) {
	t.Parallel()

	ctx := context.Background()
	options := Options{PartitionName: "test", PartitionNumber: 1, ElasticBulkSize: 100}

	t.Run("elasticsearch connection", func(t *testing.T) {
		connInfo := os.Getenv("ELASTIC_CONNECTION_INFO")
		if connInfo == "" {
			t.Skip("ELASTIC_CONNECTION_INFO not set")
		}

		var esConn types.ElasticsearchConnection
		err := json.Unmarshal([]byte(connInfo), &esConn)
		require.NoError(t, err, "failed to parse ELASTIC_CONNECTION_INFO")

		indexer, err := createVectorStoreIndexer(ctx, esConn, options)
		require.NoError(t, err)
		require.NotNil(t, indexer)
	})

	t.Run("postgresql adapter not implemented", func(t *testing.T) {
		pgConn := types.PostgreSQLConnection{Host: "localhost", Port: 5432}
		_, err := createVectorStoreIndexer(ctx, pgConn, options)
		require.Error(t, err)
		require.Contains(t, err.Error(), "indexing not implemented for adapter: postgresql")
	})

	t.Run("opensearch adapter not implemented", func(t *testing.T) {
		osConn := types.OpenSearchConnection{URL: []string{"http://localhost:9200"}}
		_, err := createVectorStoreIndexer(ctx, osConn, options)
		require.Error(t, err)
		require.Contains(t, err.Error(), "indexing not implemented for adapter: opensearch")
	})

	t.Run("unknown adapter type", func(t *testing.T) {
		_, err := createVectorStoreIndexer(ctx, mockUnknownConnection{}, options)
		require.Error(t, err)
		require.Contains(t, err.Error(), "unknown adapter: unknown")
	})
}

func TestOptionsDecoding(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name        string
		jsonInput   string
		expectError bool
		validate    func(t *testing.T, options Options)
	}{
		{
			name:        "valid JSON",
			jsonInput:   `{"project_id": 123, "from_sha": "abc", "to_sha": "def", "chunk_size": 1000, "gitaly_config": {}}`,
			expectError: false,
			validate: func(t *testing.T, options Options) {
				require.Equal(t, uint64(123), options.ProjectID)
				require.Equal(t, uint16(1000), options.ChunkSize)
			},
		},
		{
			name:        "invalid JSON syntax",
			jsonInput:   `{"project_id": 123, "invalid_json"`,
			expectError: true,
			validate:    nil,
		},
		{
			name:        "unknown fields",
			jsonInput:   `{"project_id": 123, "unknown_field": "value"}`,
			expectError: true,
			validate:    nil,
		},
		{
			name:        "wrong type for project_id",
			jsonInput:   `{"project_id": "not-a-number"}`,
			expectError: true,
			validate:    nil,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			var options Options
			decoder := json.NewDecoder(strings.NewReader(tt.jsonInput))
			decoder.DisallowUnknownFields()
			err := decoder.Decode(&options)

			if tt.expectError {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
				if tt.validate != nil {
					tt.validate(t, options)
				}
			}
		})
	}
}

func TestChunkIndexerBulkDelete(t *testing.T) {
	t.Parallel()

	ctx := context.Background()

	t.Run("successful deletion", func(t *testing.T) {
		mockIndexer := &mockIndexingStrategy{}
		chunkIndexer := &ChunkIndexer{
			options:            Options{ProjectID: 123},
			vectorStoreIndexer: mockIndexer,
		}

		paths := []string{"file1.go", "file2.go"}
		err := chunkIndexer.BulkDelete(ctx, paths)
		require.NoError(t, err)
		require.Equal(t, paths, mockIndexer.deletedPaths)
	})

	t.Run("deletion error", func(t *testing.T) {
		mockIndexer := &mockIndexingStrategy{deleteError: fmt.Errorf("deletion failed")}
		chunkIndexer := &ChunkIndexer{
			options:            Options{ProjectID: 123},
			vectorStoreIndexer: mockIndexer,
		}

		err := chunkIndexer.BulkDelete(ctx, []string{"file.go"})
		require.Error(t, err)
		require.Contains(t, err.Error(), "chunkIndexer.vectorStoreIndexer.DeletePaths")
	})
}

func TestChunkIndexerBulkIndex(t *testing.T) {
	t.Parallel()

	ctx := context.Background()
	byteConverter, _ := types.NewByteConverter(1024 * 1024)

	tests := []struct {
		name         string
		files        []gitaly.File
		chunkerError error
		indexerError error
		chunkIDs     []string
		expectError  bool
	}{
		{
			name:         "successful indexing",
			files:        []gitaly.File{{Path: "file1.go", Content: []byte("package main"), Oid: "abc123"}},
			chunkerError: nil,
			indexerError: nil,
			chunkIDs:     []string{"chunk1"},
			expectError:  false,
		},
		{
			name:         "binary file should be skipped",
			files:        []gitaly.File{{Path: "image.png", Content: []byte{0x89, 0x50, 0x4E, 0x47}, Oid: "def456"}},
			chunkerError: nil,
			indexerError: nil,
			chunkIDs:     []string{},
			expectError:  false,
		},
		{
			name:         "chunker error",
			files:        []gitaly.File{{Path: "file1.go", Content: []byte("package main"), Oid: "abc123"}},
			chunkerError: fmt.Errorf("chunking failed"),
			indexerError: nil,
			chunkIDs:     nil,
			expectError:  true,
		},
		{
			name:         "indexer error",
			files:        []gitaly.File{{Path: "file1.go", Content: []byte("package main"), Oid: "abc123"}},
			chunkerError: nil,
			indexerError: fmt.Errorf("indexing failed"),
			chunkIDs:     nil,
			expectError:  true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			mockChunker := &mockChunker{err: tt.chunkerError}
			mockIndexer := &mockIndexingStrategy{
				indexError: tt.indexerError,
				chunkIDs:   tt.chunkIDs,
			}
			mockStreamer := streamer.NewStdout()

			chunkIndexer := &ChunkIndexer{
				options:            Options{ProjectID: 123},
				chunker:            mockChunker,
				vectorStoreIndexer: mockIndexer,
				stdoutStreamer:     mockStreamer,
				byteConverter:      byteConverter,
			}

			err := chunkIndexer.BulkIndex(ctx, tt.files)

			if tt.expectError {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
		})
	}
}

func TestChunkIndexerPerformIndexing(t *testing.T) {
	t.Parallel()

	ctx := context.Background()
	byteConverter, _ := types.NewByteConverter(1024 * 1024)
	mockChunker := &mockChunker{err: nil}
	mockStreamer := streamer.NewStdout()
	var mockGitalyReader gitaly.GitalyReader

	tests := []struct {
		name       string
		reindexing bool
		fromSHA    string
	}{
		{
			name:       "normal indexing",
			reindexing: false,
			fromSHA:    "",
		},
		{
			name:       "reindexing",
			reindexing: true,
			fromSHA:    "",
		},
		{
			name:       "reindexing",
			reindexing: true,
			fromSHA:    "0000000000000000000000000000000000000000", // zero sha
		},
		{
			name:       "reindexing",
			reindexing: true,
			fromSHA:    "0000000000000000000000000000000000000000000000000000000000000000", // zero sha 256
		},
		{
			name:       "reindexing",
			reindexing: true,
			fromSHA:    "4b825dc642cb6eb9a060e54bf8d69288fbee4904", // null tree sha
		},
		{
			name:       "reindexing",
			reindexing: true,
			fromSHA:    "6ef19b41225c5369f1c104d45d8d85efa9b057b53b14b4b9b939dd74decc5321", // null tree sha 256
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			mockIndexer := &mockIndexingStrategy{
				indexError: nil,
				chunkIDs:   []string{"chunk1"},
			}

			mockGitalyReader = gitaly.NewMockGitalyReader(
				[]gitaly.File{{Path: "file1.go", Content: []byte("package main"), Oid: "abc123"}},
				[]string{"file3.go"},
			)

			options := Options{
				ProjectID:       123,
				GitalyBatchSize: 10,
				FromSHA:         tt.fromSHA,
				ForceReindex:    tt.reindexing,
			}
			chunkIndexer := &ChunkIndexer{
				options:            options,
				chunker:            mockChunker,
				stdoutStreamer:     mockStreamer,
				byteConverter:      byteConverter,
				gitalyClient:       mockGitalyReader,
				vectorStoreIndexer: mockIndexer,
			}

			err := chunkIndexer.PerformIndexing(ctx)
			require.NoError(t, err)

			require.Equal(t, tt.reindexing, mockIndexer.resolveReindexingCalled)
		})
	}
}

// Mock implementations
type mockUnknownConnection struct{}

func (m mockUnknownConnection) GetAdapterType() types.AdapterType {
	return "unknown"
}

type mockIndexingStrategy struct {
	deleteError             error
	indexError              error
	deletedPaths            []string
	chunkIDs                []string
	resolveReindexingCalled bool
}

func (m *mockIndexingStrategy) DeletePaths(ctx context.Context, projectID uint64, paths []string) error {
	m.deletedPaths = paths
	return m.deleteError
}

func (m *mockIndexingStrategy) Index(ctx context.Context, projectID uint64, chunks []chunker.Chunk) ([]string, error) {
	if m.indexError != nil {
		return nil, m.indexError
	}
	return m.chunkIDs, nil
}

func (m *mockIndexingStrategy) ResolveReindexing(ctx context.Context, projectID uint64) error {
	m.resolveReindexingCalled = true
	return nil
}

func (m *mockIndexingStrategy) Flush(ctx context.Context) ([]string, error) {
	return nil, nil
}

func (m *mockIndexingStrategy) Close(ctx context.Context) error {
	return nil
}

type mockChunker struct {
	err error
}

func (m *mockChunker) ChunkFiles(ctx context.Context, files []types.File) ([]chunker.Chunk, error) {
	if m.err != nil {
		return nil, m.err
	}

	chunks := make([]chunker.Chunk, 0)
	for _, file := range files {
		chunks = append(chunks, chunker.Chunk{
			Path:    file.Path,
			Content: file.Content,
			OID:     file.OID,
		})
	}
	return chunks, nil
}
