package indexer_test

import (
	"context"
	"path"
	"strconv"
	"testing"
	"time"

	"github.com/stretchr/testify/require"

	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/advanced/indexer"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/shared/gitaly"
)

const (
	sha                   = "9876543210987654321098765432109876543210"
	oid                   = "0123456789012345678901234567890123456789"
	parentID              = int64(667)
	parentGroupID         = int64(1)
	parentGroupIDString   = "1"
	parentIDString        = "667"
	visibilityLevel       = int8(10)
	repositoryAccessLevel = int8(20)
	wikiAccessLevel       = int8(20)
	traversalIds          = "2312-1234-2222-"
	hashedRootNamespaceId = int16(63)
	archived              = "true"
	blobSchemaVersion     = uint16(23_08)
	commitSchemaVersion   = uint16(23_06)
	wikiSchemaVersion     = uint16(23_08)
)

func setupEncoder() *indexer.Encoder {
	return indexer.NewEncoder(1024 * 1024)
}

type fakeSubmitter struct {
	flushed      int
	indexed      int
	indexedID    []string
	indexedThing []interface{}
	traversalIDs string
	removed      int
	removedID    []string
	removedType  []string
	parentID     int64
}

type fakeRepository struct {
	commits []*gitaly.Commit

	added    []*gitaly.File
	modified []*gitaly.File
	removed  []*gitaly.File
}

func (f *fakeSubmitter) ParentID() int64 {
	return f.parentID
}

func (f *fakeSubmitter) ParentGroupID() int64 {
	return parentGroupID
}

func (f *fakeSubmitter) TraversalIDs() string {
	return f.traversalIDs
}

func (f *fakeSubmitter) ProjectPermissions() *indexer.ProjectPermissions {
	projectPermissions := validProjectPermissions()
	return &projectPermissions
}

func (f *fakeSubmitter) WikiPermissions() *indexer.WikiPermissions {
	wikiPermissions := validWikiPermissions()
	return &wikiPermissions
}

func (f *fakeSubmitter) Archived() string {
	return archived
}

func (f *fakeSubmitter) HashedRootNamespaceId() int16 {
	return hashedRootNamespaceId
}

func (f *fakeSubmitter) SchemaVersionBlob() uint16 {
	return blobSchemaVersion
}

func (f *fakeSubmitter) SchemaVersionCommit() uint16 {
	return commitSchemaVersion
}

func (f *fakeSubmitter) SchemaVersionWiki() uint16 {
	return wikiSchemaVersion
}

func (f *fakeSubmitter) Index(documentType string, id string, thing interface{}) {
	f.indexed++
	f.indexedID = append(f.indexedID, id)
	f.indexedThing = append(f.indexedThing, thing)
}

func (f *fakeSubmitter) Remove(documentType, id string) {
	f.removed++
	f.removedID = append(f.removedID, id)
	f.removedType = append(f.removedType, documentType)
}

func (f *fakeSubmitter) IsProjectDocument() bool {
	return f.parentID > 0
}

func (f *fakeSubmitter) IsGroupDocument() bool {
	return !f.IsProjectDocument()
}

func (f *fakeSubmitter) Flush() error {
	f.flushed++
	return nil
}

func (r *fakeRepository) EachFileChangeBatched(ctx context.Context, bulkPut gitaly.BulkPutFunc, bulkDel gitaly.BulkDelFunc, indexBatchSize int) error {
	// Combine added and modified files
	files := make([]gitaly.File, 0, len(r.added)+len(r.modified))
	for _, file := range r.added {
		files = append(files, *file)
	}
	for _, file := range r.modified {
		files = append(files, *file)
	}

	if len(files) > 0 {
		if err := bulkPut(ctx, files); err != nil {
			return err
		}
	}

	// Handle removed files
	deletedPaths := make([]string, 0, len(r.removed))
	for _, file := range r.removed {
		deletedPaths = append(deletedPaths, file.Path)
	}

	if len(deletedPaths) > 0 {
		if err := bulkDel(ctx, deletedPaths); err != nil {
			return err
		}
	}

	return nil
}

func (r *fakeRepository) EachCommit(f gitaly.CommitFunc) error {
	for _, commit := range r.commits {
		if err := f(commit); err != nil {
			return err
		}
	}

	return nil
}
func (r *fakeRepository) GetLimitFileSize() int64 {
	return 1024 * 1024
}

func (r *fakeRepository) GetFromHash() string {
	return sha
}

func (r *fakeRepository) GetToHash() string {
	return sha
}

func setupIndexer(groupWiki bool) (*indexer.Indexer, *fakeRepository, *fakeSubmitter) {
	repo := &fakeRepository{}
	var projectId int64
	if groupWiki {
		projectId = -1
	} else {
		projectId = parentID
	}

	submitter := &fakeSubmitter{
		traversalIDs: traversalIds,
		parentID:     projectId,
	}

	return indexer.NewIndexer(repo, submitter), repo, submitter
}

func gitFile(path, content string) *gitaly.File {
	return &gitaly.File{
		Path:     path,
		Content:  []byte(content),
		TooLarge: false,
		Oid:      oid,
	}
}

func gitCommit(message string) *gitaly.Commit {
	return &gitaly.Commit{
		Author: gitaly.Signature{
			Email: "job@gitlab.com",
			Name:  "Job van der Voort",
			When:  time.Date(2016, time.September, 27, 14, 37, 46, 0, time.UTC),
		},
		Committer: gitaly.Signature{
			Email: "nick@gitlab.com",
			Name:  "Nick Thomas",
			When:  time.Date(2017, time.October, 28, 15, 38, 47, 1, time.UTC),
		},
		Message: message,
		Hash:    sha,
	}
}

func validBlob(file *gitaly.File, content string) *indexer.Blob {
	return &indexer.Blob{
		Type:      "blob",
		ID:        indexer.GenerateBlobID(parentID, file.Path),
		OID:       oid,
		RepoID:    parentIDString,
		CommitSHA: sha,
		Content:   content,
		Path:      file.Path,
		Filename:  path.Base(file.Path),
		Language:  "Text",
	}
}

func validCommit(gitCommit *gitaly.Commit) *indexer.Commit {
	return &indexer.Commit{
		Type:      "commit",
		ID:        indexer.GenerateCommitID(parentID, gitCommit.Hash),
		Author:    indexer.BuildPerson(gitCommit.Author, setupEncoder()),
		Committer: indexer.BuildPerson(gitCommit.Committer, setupEncoder()),
		RepoID:    parentIDString,
		Message:   gitCommit.Message,
		SHA:       sha,
	}
}

func validProjectPermissions() indexer.ProjectPermissions {
	return indexer.ProjectPermissions{
		VisibilityLevel:       visibilityLevel,
		RepositoryAccessLevel: repositoryAccessLevel,
	}
}

func validWikiPermissions() indexer.WikiPermissions {
	return indexer.WikiPermissions{
		VisibilityLevel: visibilityLevel,
		WikiAccessLevel: wikiAccessLevel,
	}
}

func index(idx *indexer.Indexer, blobType string) error {
	if err := idx.IndexBlobs(context.Background(), blobType); err != nil {
		return err
	}

	if err := idx.IndexCommits(); err != nil {
		return err
	}

	if err := idx.Flush(); err != nil {
		return err
	}

	return nil
}

func TestIndex(t *testing.T) {
	idx, repo, submit := setupIndexer(false)

	gitCommit := gitCommit("Initial commit")
	gitAdded := gitFile("foo/bar", "added file")
	gitModified := gitFile("foo/baz", "modified file")
	gitRemoved := gitFile("foo/qux", "removed file")

	gitTooBig := gitFile("invalid/too-big", "")
	gitTooBig.TooLarge = true

	gitBinary := gitFile("nodisplay/binary.ninja", "foo\x00")

	added := validBlob(gitAdded, "added file")

	// If the content is binary, no results (Text) will be returned. This matches the
	// behavior of Linguist.detect: https://github.com/github/linguist/blob/aad49acc0624c70d654a8dce447887dbbc713c7a/lib/linguist.rb#L14-L49
	binary := validBlob(gitBinary, indexer.NoCodeContentMsgHolder)

	modified := validBlob(gitModified, "modified file")
	removed := validBlob(gitRemoved, "removed file")
	tooBig := validBlob(gitTooBig, "")

	repo.added = append(repo.added, gitAdded, gitTooBig, gitBinary)
	repo.modified = append(repo.modified, gitModified)
	repo.removed = append(repo.removed, gitRemoved)
	repo.commits = append(repo.commits, gitCommit)

	joinDataBlob := map[string]string{"name": "blob", "parent": "project_" + parentIDString}

	err := index(idx, "blob")

	require.NoError(t, err)

	require.Equal(t, 5, submit.indexed)
	require.Equal(t, 1, submit.removed)

	require.Equal(t, parentIDString+"_"+added.Path, submit.indexedID[0])
	require.Equal(t, map[string]interface{}{"traversal_ids": traversalIds, "project_id": parentID, "blob": added, "join_field": joinDataBlob, "type": "blob", "visibility_level": visibilityLevel, "repository_access_level": repositoryAccessLevel, "archived": true, "schema_version": blobSchemaVersion}, submit.indexedThing[0])

	require.Equal(t, parentIDString+"_"+tooBig.Path, submit.indexedID[1])
	require.Equal(t, map[string]interface{}{"traversal_ids": traversalIds, "project_id": parentID, "blob": tooBig, "join_field": joinDataBlob, "type": "blob", "visibility_level": visibilityLevel, "repository_access_level": repositoryAccessLevel, "archived": true, "schema_version": blobSchemaVersion}, submit.indexedThing[1])

	require.Equal(t, parentIDString+"_"+binary.Path, submit.indexedID[2])
	require.Equal(t, map[string]interface{}{"traversal_ids": traversalIds, "project_id": parentID, "blob": binary, "join_field": joinDataBlob, "type": "blob", "visibility_level": visibilityLevel, "repository_access_level": repositoryAccessLevel, "archived": true, "schema_version": blobSchemaVersion}, submit.indexedThing[2])

	require.Equal(t, parentIDString+"_"+modified.Path, submit.indexedID[3])
	require.Equal(t, map[string]interface{}{"traversal_ids": traversalIds, "project_id": parentID, "blob": modified, "join_field": joinDataBlob, "type": "blob", "visibility_level": visibilityLevel, "repository_access_level": repositoryAccessLevel, "archived": true, "schema_version": blobSchemaVersion}, submit.indexedThing[3])

	require.Equal(t, parentIDString+"_"+removed.Path, submit.removedID[0])
	require.Equal(t, "blob", submit.removedType[0])

	require.Equal(t, 1, submit.flushed)
}

func TestTraversalIds(t *testing.T) {
	idx, repo, submit := setupIndexer(false)

	gitAdded := gitFile("foo/bar", "added file")
	added := validBlob(gitAdded, "added file")

	repo.added = append(repo.added, gitAdded)

	joinDataBlob := map[string]string{"name": "blob", "parent": "project_" + parentIDString}

	err := index(idx, "blob")

	require.NoError(t, err)

	require.Equal(t, 1, submit.indexed)
	require.Equal(t, parentIDString+"_"+added.Path, submit.indexedID[0])
	require.Equal(t, map[string]interface{}{"traversal_ids": traversalIds, "project_id": parentID, "blob": added, "join_field": joinDataBlob, "type": "blob", "visibility_level": visibilityLevel, "repository_access_level": repositoryAccessLevel, "archived": true, "schema_version": blobSchemaVersion}, submit.indexedThing[0])
	require.Equal(t, 1, submit.flushed)
}

func TestCommitIndex(t *testing.T) {
	idx, repo, submit := setupIndexer(false)

	gitCommit := gitCommit("Initial commit")
	commit := validCommit(gitCommit)

	repo.commits = append(repo.commits, gitCommit)

	err := index(idx, "blob")

	require.NoError(t, err)

	require.Equal(t, 1, submit.indexed)

	require.Equal(t, parentIDString+"_"+commit.SHA, submit.indexedID[0])
	commitMap, err := commit.ToMap()
	require.NoError(t, err)

	commitMap["visibility_level"] = visibilityLevel
	commitMap["repository_access_level"] = repositoryAccessLevel
	commitMap["hashed_root_namespace_id"] = hashedRootNamespaceId
	commitMap["schema_version"] = commitSchemaVersion
	commitMap["archived"], err = strconv.ParseBool(archived)
	require.NoError(t, err)

	require.Equal(t, commitMap, submit.indexedThing[0])

	require.Equal(t, 1, submit.flushed)
}

func TestIndexingProjectWiki(t *testing.T) {
	idx, repo, submit := setupIndexer(false)

	gitAdded := gitFile("foo/bar", "added file")
	gitRemoved := gitFile("foo/qux", "removed file")
	repo.added = append(repo.added, gitAdded)
	repo.removed = append(repo.removed, gitRemoved)
	removed := validBlob(gitRemoved, "removed file")

	err := index(idx, "wiki_blob")

	require.NoError(t, err)

	require.Equal(t, 1, submit.indexed)
	require.Equal(t, 1, submit.removed)

	indexedThing := submit.indexedThing[0].(map[string]interface{})
	require.Equal(t, wikiSchemaVersion, indexedThing["schema_version"])
	require.Equal(t, parentID, indexedThing["project_id"])
	require.Equal(t, parentGroupID, indexedThing["group_id"])
	require.True(t, indexedThing["archived"].(bool))

	require.Equal(t, "p_"+parentIDString+"_"+removed.Path, submit.removedID[0])
	require.Equal(t, "wiki_blob", submit.removedType[0])

	require.Equal(t, 1, submit.flushed)
}

func TestIndexingGroupWiki(t *testing.T) {
	idx, repo, submit := setupIndexer(true)

	gitAdded := gitFile("foo/bar", "added file")
	repo.added = append(repo.added, gitAdded)

	gitRemoved := gitFile("foo/qux", "removed file")
	repo.removed = append(repo.removed, gitRemoved)
	removed := validBlob(gitRemoved, "removed file")

	err := index(idx, "wiki_blob")

	require.NoError(t, err)

	require.Equal(t, 1, submit.indexed)

	indexedThing := submit.indexedThing[0].(map[string]interface{})

	require.Equal(t, wikiSchemaVersion, indexedThing["schema_version"])
	require.Nil(t, indexedThing["project_id"])
	require.Equal(t, parentGroupID, indexedThing["group_id"])
	require.Nil(t, indexedThing["archived"])

	require.Equal(t, "g_"+parentGroupIDString+"_"+removed.Path, submit.removedID[0])
	require.Equal(t, "wiki_blob", submit.removedType[0])

	require.Equal(t, 1, submit.flushed)
}

func TestIndexingMultipleFiles(t *testing.T) {
	idx, repo, submit := setupIndexer(false)

	gitOKFile := gitFile("ok", "")

	gitBreakingFile := gitFile("broken", "")

	repo.added = append(repo.added, gitBreakingFile, gitOKFile)

	err := index(idx, "blob")

	require.NoError(t, err)
	require.Equal(t, 2, submit.indexed) // Both files should be indexed
	require.Equal(t, 0, submit.removed)
	require.Equal(t, 1, submit.flushed)
}
