package indexer

import (
	"context"
	"fmt"
	"strconv"
	"strings"

	logkit "gitlab.com/gitlab-org/labkit/log"

	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/advanced/git"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/shared/gitaly"
)

type Submitter interface {
	ParentID() int64
	ParentGroupID() int64
	TraversalIDs() string
	HashedRootNamespaceId() int16
	SchemaVersionBlob() uint16
	SchemaVersionCommit() uint16
	SchemaVersionWiki() uint16
	Archived() string
	ProjectPermissions() *ProjectPermissions
	WikiPermissions() *WikiPermissions

	Index(documentType, id string, thing interface{})
	Remove(documentType, id string)

	IsProjectDocument() bool
	IsGroupDocument() bool

	Flush() error
}

type Indexer struct {
	git.Repository
	Submitter
	*Encoder
}

type ProjectPermissions struct { //nolint:musttag
	VisibilityLevel       int8
	RepositoryAccessLevel int8
}

type WikiPermissions struct {
	VisibilityLevel int8
	WikiAccessLevel int8
}

func NewIndexer(repository git.Repository, submitter Submitter) *Indexer {
	return &Indexer{
		Repository: repository,
		Submitter:  submitter,
		Encoder:    NewEncoder(repository.GetLimitFileSize()),
	}
}

func extractArchivedFlag(archived string) (bool, error) {
	archivedBool, err := strconv.ParseBool(archived)
	if err != nil {
		return false, fmt.Errorf("unable to parse boolean value %w", err)
	}
	return archivedBool, nil
}

func (i *Indexer) submitCommit(c *gitaly.Commit) error {
	commit := i.BuildCommit(c)
	var commitBody map[string]interface{}
	var err error
	commitBody, err = commit.ToMap()
	if err != nil {
		return fmt.Errorf("Commit %s, %w", c.Hash, err)
	}

	if i.Archived() != "" {
		archived, err := extractArchivedFlag(i.Archived())
		if err != nil {
			return err
		}
		commitBody["archived"] = archived
	}

	if permissions := i.ProjectPermissions(); permissions != nil {
		commitBody["visibility_level"] = permissions.VisibilityLevel
		commitBody["repository_access_level"] = permissions.RepositoryAccessLevel
	}

	if i.HashedRootNamespaceId() > 0 {
		commitBody["hashed_root_namespace_id"] = i.HashedRootNamespaceId()
	}

	commitBody["schema_version"] = i.SchemaVersionCommit()

	i.Index("commit", commit.ID, commitBody)
	return nil
}

func (i *Indexer) wikiSourceId() int64 {
	if i.IsGroupDocument() {
		return i.ParentGroupID()
	} else {
		return i.ParentID()
	}
}

func (i *Indexer) indexCommits() error {
	return i.EachCommit(i.submitCommit)
}

func (i *Indexer) indexRepoBlobs(ctx context.Context) error {
	return i.EachFileChangeBatched(ctx, i.submitRepoBlobsBatch, i.removeRepoBlobsBatch, 0)
}

func (i *Indexer) indexWikiBlobs(ctx context.Context) error {
	return i.EachFileChangeBatched(ctx, i.submitWikiBlobsBatch, i.removeWikiBlobsBatch, 0)
}

func (i *Indexer) Flush() error {
	return i.Submitter.Flush()
}

func (i *Indexer) IndexBlobs(ctx context.Context, blobType string) error {
	switch blobType {
	case "blob":
		return i.indexRepoBlobs(ctx)
	case "wiki_blob":
		return i.indexWikiBlobs(ctx)
	}

	return fmt.Errorf("unknown blob type: %v", blobType)
}

func (i *Indexer) IndexCommits() error {
	if err := i.indexCommits(); err != nil {
		logkit.WithError(err).Error("error while indexing commits")
		return err
	}

	return nil
}

// Batched adapter functions

func (i *Indexer) submitRepoBlobsBatch(ctx context.Context, files []gitaly.File) error {
	if len(files) == 0 {
		return nil
	}

	// Pre-calculate common values once
	toHash := i.GetToHash()
	parentID := i.ParentID()
	traversalIDs := i.TraversalIDs()
	archived := i.Archived()
	permissions := i.ProjectPermissions()
	schemaVersion := i.SchemaVersionBlob()
	isProjectDoc := i.IsProjectDocument()

	// Pre-build common index data
	joinData := map[string]string{
		"name":   "blob",
		"parent": fmt.Sprintf("project_%v", parentID),
	}

	baseIndexData := map[string]interface{}{
		"project_id":     parentID,
		"type":           "blob",
		"join_field":     joinData,
		"schema_version": schemaVersion,
	}

	// Add optional fields once
	if traversalIDs != "" {
		baseIndexData["traversal_ids"] = traversalIDs
	}

	if archived != "" {
		if archivedFlag, err := extractArchivedFlag(archived); err == nil {
			baseIndexData["archived"] = archivedFlag
		}
	}

	if permissions != nil {
		baseIndexData["visibility_level"] = permissions.VisibilityLevel
		baseIndexData["repository_access_level"] = permissions.RepositoryAccessLevel
	}

	// Process all files in batch
	for _, file := range files {
		blob, err := BuildBlob(&file, parentID, toHash, "blob", i.Encoder, isProjectDoc)
		if err != nil {
			return fmt.Errorf("Blob %s: %w", file.Path, err)
		}

		// Clone base data and add blob-specific data
		indexData := make(map[string]interface{})
		for k, v := range baseIndexData {
			indexData[k] = v
		}
		indexData["blob"] = blob

		i.Index("blob", blob.ID, indexData)
	}

	return nil
}

func (i *Indexer) removeRepoBlobsBatch(ctx context.Context, paths []string) error {
	if len(paths) == 0 {
		return nil
	}

	// Pre-calculate parentID once
	parentID := i.ParentID()

	// Process all removals in batch
	for _, path := range paths {
		blobID := GenerateBlobID(parentID, path)
		i.Remove("blob", blobID)
	}
	return nil
}

func (i *Indexer) submitWikiBlobsBatch(ctx context.Context, files []gitaly.File) error {
	if len(files) == 0 {
		return nil
	}

	// Pre-calculate common values once
	toHash := i.GetToHash()
	wikiSourceID := i.wikiSourceId()
	isProjectDoc := i.IsProjectDocument()
	archived := i.Archived()
	parentGroupID := i.ParentGroupID()
	schemaVersion := i.SchemaVersionWiki()
	traversalIDs := i.TraversalIDs()

	// Process all files in batch
	for _, file := range files {
		wikiBlob, err := BuildBlob(&file, wikiSourceID, toHash, "wiki_blob", i.Encoder, isProjectDoc)
		if err != nil {
			return fmt.Errorf("WikiBlob %s: %w", file.Path, err)
		}

		indexData, err := wikiBlob.ToMap()
		if err != nil {
			return fmt.Errorf("WikiBlob %w", err)
		}

		if isProjectDoc {
			indexData["project_id"] = i.ParentID()
			indexData["rid"] = strings.Replace(wikiBlob.RepoID, "wiki", "wiki_project", 1)
			if archived != "" {
				if archivedFlag, err := extractArchivedFlag(archived); err == nil {
					indexData["archived"] = archivedFlag
				}
			}
		}

		if parentGroupID > 0 {
			indexData["group_id"] = parentGroupID
			// Group wiki
			if i.IsGroupDocument() {
				indexData["rid"] = strings.Replace(wikiBlob.RepoID, "wiki", "wiki_group", 1)
			}
		}

		indexData["schema_version"] = schemaVersion

		if traversalIDs != "" {
			indexData["traversal_ids"] = traversalIDs
		}

		if permissions := i.WikiPermissions(); permissions != nil {
			indexData["visibility_level"] = permissions.VisibilityLevel
			indexData["wiki_access_level"] = permissions.WikiAccessLevel
		}

		i.Index("wiki_blob", wikiBlob.ID, indexData)
	}

	return nil
}

func (i *Indexer) removeWikiBlobsBatch(ctx context.Context, paths []string) error {
	if len(paths) == 0 {
		return nil
	}

	// Pre-calculate common values once
	wikiSourceID := i.wikiSourceId()
	isProjectDoc := i.IsProjectDocument()

	// Process all removals in batch
	for _, path := range paths {
		blobID := GenerateWikiBlobId(wikiSourceID, path, isProjectDoc)
		i.Remove("wiki_blob", blobID)
	}
	return nil
}
