#!/usr/bin/env bash
set -euo pipefail

# Script to check if tool versions in .tool-versions and .gitlab-ci.yml are in sync
# Automatically detects tools defined in either file and checks for version compatibility

TOOL_VERSIONS_FILE=".tool-versions"
CI_FILE=".gitlab-ci.yml"

if [ ! -f "$TOOL_VERSIONS_FILE" ]; then
  echo "Error: $TOOL_VERSIONS_FILE not found"
  exit 1
fi

if [ ! -f "$CI_FILE" ]; then
  echo "Error: $CI_FILE not found"
  exit 1
fi

# Function to check if CI version is a prefix of tool version
version_compatible() {
  local ci_version=$1
  local tool_version=$2

  # Check if ci_version is a prefix of tool_version
  if [[ "$tool_version" == "$ci_version"* ]]; then
    return 0  # true
  else
    return 1  # false
  fi
}

EXIT_CODE=0

# Check Go version
TOOL_GO_VERSION=$(grep "^golang " $TOOL_VERSIONS_FILE | awk '{print $2}' || echo "")
CI_GO_VERSION=$(grep "DEFAULT_GO_VERSION:" $CI_FILE | awk -F'"' '{print $2}' || echo "")

if [[ -n "$TOOL_GO_VERSION" && -n "$CI_GO_VERSION" ]]; then
  echo "Checking Go versions:"
  echo "  .gitlab-ci.yml: $CI_GO_VERSION"
  echo "  .tool-versions: $TOOL_GO_VERSION"

  if ! version_compatible "$CI_GO_VERSION" "$TOOL_GO_VERSION"; then
    echo "❌ Error: Go versions are not in sync"
    echo "   CI version ($CI_GO_VERSION) is not a prefix of tool version ($TOOL_GO_VERSION)"
    EXIT_CODE=1
  else
    echo "✅ Go versions are compatible"
  fi
elif [[ -n "$TOOL_GO_VERSION" ]]; then
  echo "⚠️  Warning: Go version defined in .tool-versions but not in .gitlab-ci.yml"
elif [[ -n "$CI_GO_VERSION" ]]; then
  echo "⚠️  Warning: Go version defined in .gitlab-ci.yml but not in .tool-versions"
fi

# Check golangci-lint version
TOOL_GOLANGCI_VERSION=$(grep "^golangci-lint " $TOOL_VERSIONS_FILE | awk '{print $2}' || echo "")
CI_GOLANGCI_VERSION=$(grep "GOLANGCI_VERSION:" $CI_FILE | awk -F'"' '{print $2}' || echo "")

if [[ -n "$TOOL_GOLANGCI_VERSION" && -n "$CI_GOLANGCI_VERSION" ]]; then
  echo -e "\nChecking golangci-lint versions:"
  echo "  .gitlab-ci.yml: $CI_GOLANGCI_VERSION"
  echo "  .tool-versions: $TOOL_GOLANGCI_VERSION"

  if [ "$CI_GOLANGCI_VERSION" != "$TOOL_GOLANGCI_VERSION" ]; then
    echo "❌ Error: golangci-lint versions do not match"
    echo "   CI version: $CI_GOLANGCI_VERSION"
    echo "   Tool version: $TOOL_GOLANGCI_VERSION"
    EXIT_CODE=1
  else
    echo "✅ golangci-lint versions match"
  fi
elif [[ -n "$TOOL_GOLANGCI_VERSION" ]]; then
  echo -e "\n⚠️  Warning: golangci-lint version defined in .tool-versions but not in .gitlab-ci.yml"
elif [[ -n "$CI_GOLANGCI_VERSION" ]]; then
  echo -e "\n⚠️  Warning: golangci-lint version defined in .gitlab-ci.yml but not in .tool-versions"
fi

# Add more tool checks here if needed in the future
# Example:
# Check for TOOL_NAME by adding:
# TOOL_VERSION=$(grep "^tool_name " $TOOL_VERSIONS_FILE | awk '{print $2}' || echo "")
# CI_VERSION=$(grep "CI_VAR_NAME:" $CI_FILE | awk -F'"' '{print $2}' || echo "")

exit $EXIT_CODE
