#!/usr/bin/env bash
set -euo pipefail

# Script to validate that CHANGELOG.md was generated from the correct base commit
# This ensures that changelog updates in MRs are based on the current main branch

# Check if we're in a merge request
if [ -z "${CI_MERGE_REQUEST_DIFF_BASE_SHA:-}" ]; then
  echo "Not in a merge request context, skipping validation"
  exit 0
fi

# Check if CHANGELOG.md has been modified
if ! git diff --name-only "$CI_MERGE_REQUEST_DIFF_BASE_SHA" HEAD | grep -q "^CHANGELOG.md$"; then
  echo "✓ CHANGELOG.md not modified, skipping validation"
  exit 0
fi

echo "CHANGELOG.md was modified, validating metadata..."

# Get the commit SHA from the current branch's CHANGELOG.md
CURRENT_SHA=$(./scripts/changelog-metadata.sh get CHANGELOG.md commit_sha 2>/dev/null || echo "")

if [ -z "$CURRENT_SHA" ]; then
  echo "⚠ Warning: No metadata found in CHANGELOG.md"
  echo "This is expected if this MR is adding the metadata feature."
  exit 0
fi

echo "CHANGELOG.md metadata shows it was generated from commit: $CURRENT_SHA"

# Check if the SHA matches the base of the MR
if [ "$CURRENT_SHA" = "$CI_MERGE_REQUEST_DIFF_BASE_SHA" ]; then
  echo "✓ CHANGELOG.md was correctly generated from the base commit"
  exit 0
fi

# The SHA doesn't match - check if it's an ancestor (could be from an older base)
if git merge-base --is-ancestor "$CURRENT_SHA" "$CI_MERGE_REQUEST_DIFF_BASE_SHA" 2>/dev/null; then
  echo "✗ CHANGELOG.md was generated from an older commit"
  echo "  Generated from: $CURRENT_SHA"
  echo "  Expected:       $CI_MERGE_REQUEST_DIFF_BASE_SHA (current base)"
  echo ""
  echo "The CHANGELOG.md needs to be regenerated from the current main branch."
  echo "This usually happens when main has been updated since the CHANGELOG was generated."
  exit 1
fi

# The SHA is not an ancestor - this might be a different branch
echo "⚠ Warning: CHANGELOG.md was generated from a different commit tree"
echo "  Generated from: $CURRENT_SHA"
echo "  MR base:        $CI_MERGE_REQUEST_DIFF_BASE_SHA"
echo ""
echo "Please ensure the CHANGELOG.md was generated from the correct base."
exit 1
