#!/bin/sh
#
# Script to manage changelog metadata in CHANGELOG.md
#
# Usage:
#   Update metadata:
#     ./scripts/changelog-metadata.sh update [changelog_file] [commit_sha]
#
#   Extract metadata:
#     ./scripts/changelog-metadata.sh get [changelog_file] [field]
#
# Examples:
#   # Update metadata (uses CI_COMMIT_SHA if not provided)
#   ./scripts/changelog-metadata.sh update CHANGELOG.md abc123def456
#
#   # Display all metadata
#   ./scripts/changelog-metadata.sh get
#
#   # Extract specific field
#   ./scripts/changelog-metadata.sh get CHANGELOG.md commit_sha

set -euo pipefail

COMMAND="${1:-}"
CHANGELOG_FILE="${2:-CHANGELOG.md}"

usage() {
  cat << EOF
Usage: $0 <command> [options]

Commands:
  update [file] [sha]    Update changelog metadata
                         file: Path to CHANGELOG.md (default: CHANGELOG.md)
                         sha:  Commit SHA (default: \$CI_COMMIT_SHA)
  
  get [file] [field]     Extract changelog metadata
                         file:  Path to CHANGELOG.md (default: CHANGELOG.md)
                         field: Specific field to extract (optional)

Examples:
  $0 update CHANGELOG.md abc123
  $0 get
  $0 get CHANGELOG.md commit_sha
EOF
  exit 1
}

update_metadata() {
  CHANGELOG_FILE="$1"
  COMMIT_SHA="${2:-${CI_COMMIT_SHA:-}}"

  if [ -z "$COMMIT_SHA" ]; then
    echo "Error: commit SHA required (provide as argument or set CI_COMMIT_SHA)" >&2
    exit 1
  fi

  if [ ! -f "$CHANGELOG_FILE" ]; then
    echo "Error: $CHANGELOG_FILE not found" >&2
    exit 1
  fi

  # Check if placeholder exists
  if ! grep -q '<!-- changelog_metadata_placeholder -->' "$CHANGELOG_FILE"; then
    echo "Error: metadata placeholder not found in $CHANGELOG_FILE" >&2
    exit 1
  fi

  # Generate metadata
  CHANGELOG_TIMESTAMP=$(date +%s)
  CHANGELOG_ISO=$(date -u +"%Y-%m-%dT%H:%M:%SZ")

  # Create metadata JSON (compact format)
  METADATA_JSON=$(jq -nc \
    --arg sha "$COMMIT_SHA" \
    --arg timestamp "$CHANGELOG_TIMESTAMP" \
    --arg iso "$CHANGELOG_ISO" \
    '{generated_at: $timestamp|tonumber, generated_at_iso: $iso, commit_sha: $sha}')

  # Replace placeholder with actual metadata
  sed -i.bak "s|<!-- changelog_metadata_placeholder -->|<!-- changelog_metadata: $METADATA_JSON -->|" "$CHANGELOG_FILE"
  rm -f "${CHANGELOG_FILE}.bak"

  echo "✓ Metadata updated successfully"
  echo "  Commit SHA: $COMMIT_SHA"
  echo "  Timestamp: $CHANGELOG_ISO"
}

get_metadata() {
  CHANGELOG_FILE="$1"
  FIELD="${2:-}"

  if [ ! -f "$CHANGELOG_FILE" ]; then
    echo "Error: $CHANGELOG_FILE not found" >&2
    exit 1
  fi

  # Extract metadata JSON from CHANGELOG
  METADATA_LINE=$(grep -o '<!-- changelog_metadata: {.*} -->' "$CHANGELOG_FILE" 2>/dev/null || echo "")

  if [ -z "$METADATA_LINE" ]; then
    echo "Error: No changelog metadata found in $CHANGELOG_FILE" >&2
    echo "The metadata will be added on the next automated release." >&2
    exit 1
  fi

  # Extract JSON from the comment
  METADATA_JSON=$(echo "$METADATA_LINE" | sed -n 's/<!-- changelog_metadata: \(.*\) -->/\1/p')

  # If no field specified, display all metadata
  if [ -z "$FIELD" ]; then
    echo "$METADATA_JSON" | jq '.'
    exit 0
  fi

  # Extract specific field
  VALUE=$(echo "$METADATA_JSON" | jq -r ".$FIELD")

  if [ "$VALUE" = "null" ]; then
    echo "Error: Field '$FIELD' not found in metadata" >&2
    exit 1
  fi

  echo "$VALUE"
}

# Main command dispatcher
case "$COMMAND" in
  update)
    update_metadata "$CHANGELOG_FILE" "${3:-}"
    ;;
  get)
    get_metadata "$CHANGELOG_FILE" "${3:-}"
    ;;
  "")
    echo "Error: command required" >&2
    usage
    ;;
  *)
    echo "Error: unknown command '$COMMAND'" >&2
    usage
    ;;
esac
