package gitaly

import (
	"context"
	"encoding/json"
	"fmt"
	"os"
	"sort"
	"strings"
	"testing"

	"github.com/stretchr/testify/require"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	gitalyClient "gitlab.com/gitlab-org/gitaly/v18/client"
	pb "gitlab.com/gitlab-org/gitaly/v18/proto/go/gitalypb"
)

const (
	projectID       = 72724
	headSHA         = "b83d6e391c22777fca1ed3012fce84f633d7fed0"
	testRepo        = "test-gitlab-elasticsearch-indexer-shared/gitlab-test.git"
	testProjectPath = "test-gitlab-elasticsearch-indexer-shared/gitlab-test"
	testRepoPath    = "https://gitlab.com/gitlab-org/gitlab-test.git"
)

type gitalyConnectionInfo struct {
	Address string `json:"address"`
	Storage string `json:"storage"`
}

var (
	gitalyConnInfo *gitalyConnectionInfo
)

func init() {
	gci, exists := os.LookupEnv("GITALY_CONNECTION_INFO")
	if exists {
		err := json.Unmarshal([]byte(gci), &gitalyConnInfo)
		if err != nil {
			panic(err)
		}
	}
}

func ensureGitalyRepository() error {
	conn, err := gitalyClient.Dial(gitalyConnInfo.Address, gitalyClient.DefaultDialOpts)
	if err != nil {
		return fmt.Errorf("did not connect: %w", err)
	}

	repositoryClient := pb.NewRepositoryServiceClient(conn)
	repository := &pb.Repository{
		StorageName:  gitalyConnInfo.Storage,
		RelativePath: testRepo,
	}

	// Remove the repository if it already exists, for consistency
	if repositoryExistsResponse, repositoryExistsErr := repositoryClient.RepositoryExists(context.Background(), &pb.RepositoryExistsRequest{
		Repository: repository,
	}); repositoryExistsErr != nil {
		return repositoryExistsErr
	} else if repositoryExistsResponse.Exists {
		if _, err = repositoryClient.RemoveRepository(context.Background(), &pb.RemoveRepositoryRequest{
			Repository: repository,
		}); err != nil {
			removeRepositoryStatus, ok := status.FromError(err)
			if !ok || removeRepositoryStatus.Code() != codes.NotFound || removeRepositoryStatus.Message() != "repository does not exist" {
				return fmt.Errorf("remove repository: %w", err)
			}
		}
	}

	glRepository := &pb.Repository{
		StorageName:   gitalyConnInfo.Storage,
		RelativePath:  testRepo,
		GlProjectPath: testProjectPath,
	}

	createReq := &pb.CreateRepositoryFromURLRequest{
		Repository: glRepository,
		Url:        testRepoPath,
	}

	_, err = repositoryClient.CreateRepositoryFromURL(context.Background(), createReq)
	if err != nil {
		return err
	}

	writeHeadReq := &pb.WriteRefRequest{
		Repository: glRepository,
		Ref:        []byte("refs/heads/master"),
		Revision:   []byte(headSHA),
	}

	_, err = repositoryClient.WriteRef(context.Background(), writeHeadReq)
	return err
}

func checkDeps(t *testing.T) {
	if os.Getenv("GITALY_CONNECTION_INFO") == "" {
		t.Skip("GITALY_CONNECTION_INFO is not set")
	}
}

func buildConfigFromEnv() *StorageConfig {
	data := strings.NewReader(os.Getenv("GITALY_CONNECTION_INFO"))

	config := StorageConfig{
		RelativePath: testRepo,
		ProjectPath:  testProjectPath,
	}

	json.NewDecoder(data).Decode(&config)

	return &config
}

func setupTestRepository(t *testing.T) {
	checkDeps(t)
	err := ensureGitalyRepository()
	require.NoError(t, err)
}

func createGitalyClient(fromSHA string, toSHA string) *GitalyClient {
	testConfig := buildConfigFromEnv()
	client, _ := NewGitalyClient(context.TODO(), testConfig, projectID, fromSHA, toSHA, (1024 * 1024))

	return client
}

// helper function tests

func TestIsBlankSHA(t *testing.T) {
	require.True(t, IsBlankSHA(""))
	require.True(t, IsBlankSHA("0000000000000000000000000000000000000000"))
	require.True(t, IsBlankSHA("0000000000000000000000000000000000000000000000000000000000000000"))
	require.True(t, IsBlankSHA("4b825dc642cb6eb9a060e54bf8d69288fbee4904"))
	require.True(t, IsBlankSHA("6ef19b41225c5369f1c104d45d8d85efa9b057b53b14b4b9b939dd74decc5321"))
	require.False(t, IsBlankSHA("abc123"))
}

// NewGitalyClient tests

func TestNewGitalyClient(t *testing.T) {
	setupTestRepository(t)

	testConfig := buildConfigFromEnv()
	initialSHA := "1a0b36b3cdad1d2ee32457c102a8c0b7056fa863"
	client, err := NewGitalyClient(context.TODO(), testConfig, projectID, initialSHA, headSHA, (1024 * 1024))

	require.NotNil(t, client)
	require.NoError(t, err)
}

func TestNewGitalyClientSHAs(t *testing.T) {
	setupTestRepository(t)

	var client *GitalyClient

	// Test client with fromSHA="" and toSHA=""
	client = createGitalyClient("", "")
	require.Equal(t, headSHA, client.ToHash)
	require.Equal(t, NullTreeSHA, client.FromHash)

	// Test client with fromSHA=ZeroSHA
	client = createGitalyClient(ZeroSHA, headSHA)
	require.Equal(t, NullTreeSHA, client.FromHash)

	// Test client with fromSHA=ZeroSHA256
	client = createGitalyClient(ZeroSHA256, headSHA)
	require.Equal(t, NullTreeSHA256, client.FromHash)
}

// EachFileChange tests

var testRepoFiles = []string{
	".gitattributes",
	".gitignore",
	".gitmodules",
	"CHANGELOG",
	"CONTRIBUTING.md",
	"Gemfile.zip",
	"LICENSE",
	"MAINTENANCE.md",
	"PROCESS.md",
	"README",
	"README.md",
	"VERSION",
	"bar/branch-test.txt",
	"custom-highlighting/test.gitlab-custom",
	"encoding/feature-1.txt",
	"encoding/feature-2.txt",
	"encoding/hotfix-1.txt",
	"encoding/hotfix-2.txt",
	"encoding/iso8859.txt",
	"encoding/russian.rb",
	"encoding/test.txt",
	"encoding/テスト.txt",
	"encoding/テスト.xls",
	"files/html/500.html",
	"files/images/6049019_460s.jpg",
	"files/images/logo-black.png",
	"files/images/logo-white.png",
	"files/images/wm.svg",
	"files/js/application.js",
	"files/js/commit.coffee",
	"files/lfs/lfs_object.iso",
	"files/markdown/ruby-style-guide.md",
	"files/ruby/popen.rb",
	"files/ruby/regex.rb",
	"files/ruby/version_info.rb",
	"files/whitespace",
	"foo/bar/.gitkeep",
	"with space/README.md",
}

func runEachFileChangeBatched(t *testing.T, fromSHA string, toSHA string) ([]File, []string, error) {
	setupTestRepository(t)

	// setup bulk update function
	updatedFiles := []File{}
	bulkPutFunc := func(ctx context.Context, gitFiles []File) error {
		updatedFiles = append(updatedFiles, gitFiles...)

		return nil
	}

	// setup bulk delete function
	deletedPaths := []string{}
	bulkDelFunc := func(ctx context.Context, paths []string) error {
		deletedPaths = append(deletedPaths, paths...)

		return nil
	}

	client := createGitalyClient(fromSHA, toSHA)
	err := client.EachFileChangeBatched(context.TODO(), bulkPutFunc, bulkDelFunc, 0)
	if err != nil {
		return nil, nil, err
	}

	return updatedFiles, deletedPaths, nil
}

func TestEachFileChangeAllCommits(t *testing.T) {
	updatedFiles, deletedPaths, err := runEachFileChangeBatched(t, "", headSHA)

	require.NoError(t, err)

	// Check all deleted paths
	require.Equal(t, []string{}, deletedPaths)

	// Check updated files
	updatedFilePaths := []string{}
	var versionFile File
	for _, file := range updatedFiles {
		updatedFilePaths = append(updatedFilePaths, file.Path)

		// get the VERSION file for testing a single file
		if file.Path == "VERSION" {
			versionFile = file
		}
	}

	sort.Strings(updatedFilePaths)
	require.Equal(t, testRepoFiles, updatedFilePaths)

	// Check version file in detail
	require.NoError(t, err)
	require.Equal(t, "998707b421c89bd9a3063333f9f728ef3e43d101", versionFile.Oid)
	require.Equal(t, "6.7.0.pre\n", string(versionFile.Content))
}

func TestEachFileChangeInCommitRange(t *testing.T) {
	// For a visual comparison of this commit range, please see:
	// https://gitlab.com/gitlab-org/gitlab-test/-/compare/1b12f15a11fc6e62177bef08f47bc7b5ce50b141...e63f41fe459e62e1228fcef60d7189127aeba95a?from_project_id=72724
	updatedFiles, deletedPaths, err := runEachFileChangeBatched(t, "1b12f15a11fc6e62177bef08f47bc7b5ce50b141", "e63f41fe459e62e1228fcef60d7189127aeba95a")

	require.NoError(t, err)

	// We expect no deletions for this commit range
	require.Equal(t, []string{}, deletedPaths)

	// We expect 2 updated files for this commit range
	updatedFilePaths := []string{}
	for _, file := range updatedFiles {
		updatedFilePaths = append(updatedFilePaths, file.Path)
	}

	expectedUpdatedFiles := []string{"README.md", "bar/branch-test.txt"}
	sort.Strings(updatedFilePaths)
	require.Equal(t, expectedUpdatedFiles, updatedFilePaths)
}

func TestEachFileChangeWithDeletions(t *testing.T) {
	// For a visual comparison of this commit range, please see:
	// https://gitlab.com/gitlab-org/gitlab-test/-/compare/281d3a76f31c812dbf48abce82ccf6860adedd81...d59c60028b053793cecfb4022de34602e1a9218e?from_project_id=72724
	updatedFiles, deletedPaths, err := runEachFileChangeBatched(t, "281d3a76f31c812dbf48abce82ccf6860adedd81", "d59c60028b053793cecfb4022de34602e1a9218e")

	require.NoError(t, err)

	// We expect 1 deleted file for this commit range
	require.Equal(t, []string{"files/js/commit.js.coffee"}, deletedPaths)

	// We expected no updated files for this commit range
	require.Equal(t, []File{}, updatedFiles)
}

func TestEachFileChangeWithRename(t *testing.T) {
	// For a visual comparison of this commit range, please see:
	// https://gitlab.com/gitlab-org/gitlab-test/-/compare/281d3a76f31c812dbf48abce82ccf6860adedd81...6907208d755b60ebeacb2e9dfea74c92c3449a1f?from_project_id=72724
	updatedFiles, deletedPaths, err := runEachFileChangeBatched(t, "281d3a76f31c812dbf48abce82ccf6860adedd81", "6907208d755b60ebeacb2e9dfea74c92c3449a1f")

	require.NoError(t, err)

	// We expect 1 deleted file for this commit range
	require.Equal(t, []string{"files/js/commit.js.coffee"}, deletedPaths)

	// We expect 1 added file for this commit range
	require.Len(t, updatedFiles, 1)
	require.Equal(t, "files/js/commit.coffee", updatedFiles[0].Path)
}
