package chunk

import (
	"encoding/json"
	"flag"
	"fmt"
	"strings"

	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/types"
)

var validAdapters = [...]types.AdapterType{
	types.PostgreSQLAdapter,
	types.OpenSearchAdapter,
	types.ElasticsearchAdapter,
}

// CommandOptions contains all the parsed command-line flags
type CommandOptions struct {
	AdapterType    types.AdapterType
	OptionsJSON    string
	ConnectionJSON string
}

func decodeConnection[T any](connectionJSON string, adapterType types.AdapterType) (T, error) {
	var result T
	decoder := json.NewDecoder(strings.NewReader(connectionJSON))
	decoder.DisallowUnknownFields()

	if err := decoder.Decode(&result); err != nil {
		return result, fmt.Errorf("failed to parse %s connection: %w", adapterType, err)
	}

	return result, nil
}

// parseConnection parses the connection JSON string based on the adapter type
func parseConnection(adapterType types.AdapterType, connectionJSON string) (types.Connection, error) {
	switch adapterType {
	case types.PostgreSQLAdapter:
		result, err := decodeConnection[types.PostgreSQLConnection](connectionJSON, adapterType)
		if err != nil {
			return nil, err
		}
		return result, nil

	case types.OpenSearchAdapter:
		result, err := decodeConnection[types.OpenSearchConnection](connectionJSON, adapterType)
		if err != nil {
			return nil, err
		}
		return result, nil

	case types.ElasticsearchAdapter:
		result, err := decodeConnection[types.ElasticsearchConnection](connectionJSON, adapterType)
		if err != nil {
			return nil, err
		}
		return result, nil

	default:
		return nil, fmt.Errorf("unknown adapter: %s", adapterType)
	}
}

func isValidAdapter(adapter types.AdapterType) bool {
	for _, validAdapter := range validAdapters {
		if adapter == validAdapter {
			return true
		}
	}
	return false
}

// parseFlags parses command line flags and validates them
func parseFlags() (CommandOptions, error) {
	var opts CommandOptions
	var adapterStr string

	flag.StringVar(&opts.OptionsJSON, "options", "", "JSON string of options for chunking")

	validAdaptersList := make([]string, len(validAdapters))
	for i, adapter := range validAdapters {
		validAdaptersList[i] = string(adapter)
	}
	flag.StringVar(&adapterStr, "adapter", "", fmt.Sprintf("Adapter to use (%s)", strings.Join(validAdaptersList, ", ")))
	flag.StringVar(&opts.ConnectionJSON, "connection", "", "JSON string of connection parameters")
	flag.Parse()

	if adapterStr == "" {
		return CommandOptions{}, fmt.Errorf("-adapter flag is required")
	}

	if opts.OptionsJSON == "" {
		return CommandOptions{}, fmt.Errorf("-options flag is required")
	}

	if opts.ConnectionJSON == "" {
		return CommandOptions{}, fmt.Errorf("-connection flag is required")
	}

	opts.AdapterType = types.AdapterType(adapterStr)
	if !isValidAdapter(opts.AdapterType) {
		return CommandOptions{}, fmt.Errorf("invalid adapter: %s, must be one of: %s", opts.AdapterType, strings.Join(validAdaptersList, ", "))
	}

	return opts, nil
}
