package postgresql

import (
	"context"
	"fmt"
	"log/slog"

	"github.com/jackc/pgx/v5/pgxpool"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/types"
)

type PostgreSQLClient struct {
	Conn types.PostgreSQLConnection
	pool *pgxpool.Pool
}

func New(conn types.PostgreSQLConnection) *PostgreSQLClient {
	return &PostgreSQLClient{
		Conn: conn,
	}
}

func (c *PostgreSQLClient) Connect(ctx context.Context) error {
	slog.Debug("connecting to PostgreSQL",
		"host", c.Conn.Host,
		"port", c.Conn.Port,
		"database", c.Conn.Database,
		"table", c.Conn.Table,
	)

	connString := fmt.Sprintf(
		"postgres://%s:%s@%s:%d/%s",
		c.Conn.User,
		c.Conn.Password,
		c.Conn.Host,
		c.Conn.Port,
		c.Conn.Database,
	)

	pool, err := pgxpool.New(ctx, connString)
	if err != nil {
		return fmt.Errorf("failed to connect to PostgreSQL: %w", err)
	}

	// Verify connection with a ping
	if err := pool.Ping(ctx); err != nil {
		return fmt.Errorf("failed to ping PostgreSQL: %w", err)
	}

	c.pool = pool

	slog.Info("PostgreSQL connection successful",
		"host", c.Conn.Host,
		"port", c.Conn.Port,
	)

	return nil
}

func (c *PostgreSQLClient) Close() {
	if c.pool != nil {
		c.pool.Close()
		slog.Debug("PostgreSQL connection closed")
	}
}

func (c *PostgreSQLClient) Pool() *pgxpool.Pool {
	return c.pool
}
