package git

import (
	"testing"

	"github.com/stretchr/testify/require"

	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/shared/gitaly"
)

func TestGitalyConnectionInfo_ToGitalyConfig(t *testing.T) {
	tests := []struct {
		name     string
		gci      GitalyConnectionInfo
		expected gitaly.StorageConfig
		fileSize int64
	}{
		{
			name: "converts all fields correctly",
			gci: GitalyConnectionInfo{
				Address:       "tcp://gitaly.example.com:8075",
				Token:         "secret-token",
				StorageName:   "default",
				RelativePath:  "group/project.git",
				ProjectPath:   "group/project",
				LimitFileSize: 1048576,
				TokenVersion:  1,
			},
			expected: gitaly.StorageConfig{
				Address:      "tcp://gitaly.example.com:8075",
				Token:        "secret-token",
				StorageName:  "default",
				RelativePath: "group/project.git",
				ProjectPath:  "group/project",
				TokenVersion: 1,
			},
			fileSize: 1048576,
		},
		{
			name: "handles empty values",
			gci: GitalyConnectionInfo{
				Address:       "",
				Token:         "",
				StorageName:   "",
				RelativePath:  "",
				ProjectPath:   "",
				LimitFileSize: 0,
				TokenVersion:  0,
			},
			expected: gitaly.StorageConfig{
				Address:      "",
				Token:        "",
				StorageName:  "",
				RelativePath: "",
				ProjectPath:  "",
				TokenVersion: 0,
			},
			fileSize: 0,
		},
		{
			name: "handles different token versions",
			gci: GitalyConnectionInfo{
				Address:       "unix:///var/opt/gitlab/gitaly/gitaly.socket",
				Token:         "v2-token",
				StorageName:   "storage-1",
				RelativePath:  "@hashed/ab/cd/abcd.git",
				ProjectPath:   "hashed-project",
				LimitFileSize: 2097152,
				TokenVersion:  2,
			},
			expected: gitaly.StorageConfig{
				Address:      "unix:///var/opt/gitlab/gitaly/gitaly.socket",
				Token:        "v2-token",
				StorageName:  "storage-1",
				RelativePath: "@hashed/ab/cd/abcd.git",
				ProjectPath:  "hashed-project",
				TokenVersion: 2,
			},
			fileSize: 2097152,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			config, fileSize := tt.gci.ToGitalyConfig()

			require.Equal(t, &tt.expected, config)
			require.Equal(t, tt.fileSize, fileSize)
		})
	}
}
