package chunk

import (
	"flag"
	"os"
	"testing"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/types"
)

func TestIsValidAdapter(t *testing.T) {
	t.Parallel()

	require.True(t, isValidAdapter(types.PostgreSQLAdapter))
	require.True(t, isValidAdapter(types.OpenSearchAdapter))
	require.True(t, isValidAdapter(types.ElasticsearchAdapter))
	require.False(t, isValidAdapter("invalid"))
	require.False(t, isValidAdapter(""))
}

func TestParseConnection(t *testing.T) {
	t.Parallel()

	// Valid connections for each adapter type
	pgConn, err := parseConnection(types.PostgreSQLAdapter, `{"host": "localhost", "port": 5432}`)
	require.NoError(t, err)
	require.Equal(t, types.PostgreSQLAdapter, pgConn.GetAdapterType())

	osConn, err := parseConnection(types.OpenSearchAdapter, `{"url": ["http://localhost:9200"]}`)
	require.NoError(t, err)
	require.Equal(t, types.OpenSearchAdapter, osConn.GetAdapterType())

	// Empty JSON should work for opensearch/elasticsearch
	esConn, err := parseConnection(types.ElasticsearchAdapter, `{}`)
	require.NoError(t, err)
	require.Equal(t, types.ElasticsearchAdapter, esConn.GetAdapterType())

	// Error cases
	_, err = parseConnection("unknown", `{}`)
	require.Error(t, err)
	require.Contains(t, err.Error(), "unknown adapter")

	_, err = parseConnection(types.PostgreSQLAdapter, `{"invalid_field": "value"}`)
	require.Error(t, err)

	_, err = parseConnection(types.PostgreSQLAdapter, `{"host": "localhost", "port": 5432,}`)
	require.Error(t, err)
}

func TestDecodeConnection(t *testing.T) {
	t.Parallel()

	// Valid JSON with all fields
	result, err := decodeConnection[types.PostgreSQLConnection](`{"host": "localhost", "port": 5432, "user": "postgres", "password": "password", "database": "gitlab", "table": "embeddings"}`, types.PostgreSQLAdapter)
	require.NoError(t, err)
	require.Equal(t, "localhost", result.Host)
	require.Equal(t, uint16(5432), result.Port)
	require.Equal(t, "postgres", result.User)

	// Partial JSON (missing some fields) - should still work
	result, err = decodeConnection[types.PostgreSQLConnection](`{"host": "localhost", "port": 5432}`, types.PostgreSQLAdapter)
	require.NoError(t, err)
	require.Equal(t, "localhost", result.Host)
	require.Equal(t, uint16(5432), result.Port)

	// Empty JSON should work
	result, err = decodeConnection[types.PostgreSQLConnection](`{}`, types.PostgreSQLAdapter)
	require.NoError(t, err)

	// Unknown fields should fail (DisallowUnknownFields)
	_, err = decodeConnection[types.PostgreSQLConnection](`{"host": "localhost", "port": 5432, "unknown_field": "value"}`, types.PostgreSQLAdapter)
	require.Error(t, err)

	// Invalid JSON syntax should fail
	_, err = decodeConnection[types.PostgreSQLConnection](`{"host": "localhost", "port": }`, types.PostgreSQLAdapter)
	require.Error(t, err)
}

// Helper function to temporarily set command line args and restore them after the test
func withCommandLineArgs(args []string, testFunc func()) {
	oldArgs := os.Args
	oldFlagCommandLine := flag.CommandLine

	os.Args = args
	flag.CommandLine = flag.NewFlagSet(args[0], flag.ExitOnError)

	testFunc()

	os.Args = oldArgs
	flag.CommandLine = oldFlagCommandLine
}

func TestParseFlags(t *testing.T) {
	// Valid flags
	withCommandLineArgs([]string{"program", "-adapter", "postgresql", "-options", "{}", "-connection", "{}"}, func() {
		opts, err := parseFlags()
		require.NoError(t, err)
		require.Equal(t, types.PostgreSQLAdapter, opts.AdapterType)
		require.Equal(t, "{}", opts.OptionsJSON)
		require.Equal(t, "{}", opts.ConnectionJSON)
	})

	// Missing required flags
	withCommandLineArgs([]string{"program", "-options", "{}", "-connection", "{}"}, func() {
		_, err := parseFlags()
		require.Error(t, err)
		require.Contains(t, err.Error(), "-adapter flag is required")
	})

	withCommandLineArgs([]string{"program", "-adapter", "postgresql", "-connection", "{}"}, func() {
		_, err := parseFlags()
		require.Error(t, err)
		require.Contains(t, err.Error(), "-options flag is required")
	})

	withCommandLineArgs([]string{"program", "-adapter", "postgresql", "-options", "{}"}, func() {
		_, err := parseFlags()
		require.Error(t, err)
		require.Contains(t, err.Error(), "-connection flag is required")
	})

	// Invalid adapter
	withCommandLineArgs([]string{"program", "-adapter", "invalid", "-options", "{}", "-connection", "{}"}, func() {
		_, err := parseFlags()
		require.Error(t, err)
		require.Contains(t, err.Error(), "invalid adapter")
	})
}
