package advanced

import (
	"os"
	"testing"

	"github.com/stretchr/testify/require"

	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/advanced/git"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/advanced/indexer"
)

func TestReadConfig(t *testing.T) {
	tests := []struct {
		name           string
		envValue       string
		repoPath       string
		projectPath    string
		expectedError  bool
		expectedConfig *git.GitalyConnectionInfo
	}{
		{
			name:        "valid JSON config",
			envValue:    `{"address":"tcp://gitaly:8075","token":"secret","storage":"default"}`,
			repoPath:    "group/project.git",
			projectPath: "group/project",
			expectedConfig: &git.GitalyConnectionInfo{
				Address:       "tcp://gitaly:8075",
				Token:         "secret",
				StorageName:   "default",
				RelativePath:  "group/project.git",
				ProjectPath:   "group/project",
				LimitFileSize: defaultLimitFileSize,
			},
		},
		{
			name:        "empty JSON config",
			envValue:    `{}`,
			repoPath:    "test.git",
			projectPath: "test",
			expectedConfig: &git.GitalyConnectionInfo{
				RelativePath:  "test.git",
				ProjectPath:   "test",
				LimitFileSize: defaultLimitFileSize,
			},
		},
		{
			name:          "invalid JSON",
			envValue:      `{invalid-json}`,
			repoPath:      "test.git",
			projectPath:   "test",
			expectedError: true,
		},
		{
			name:        "config with custom file size",
			envValue:    `{"address":"unix:///var/gitaly.socket","limit_file_size":2097152}`,
			repoPath:    "project.git",
			projectPath: "project",
			expectedConfig: &git.GitalyConnectionInfo{
				Address:       "unix:///var/gitaly.socket",
				RelativePath:  "project.git",
				ProjectPath:   "project",
				LimitFileSize: 2097152,
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			originalEnv := os.Getenv("GITALY_CONNECTION_INFO")
			defer os.Setenv("GITALY_CONNECTION_INFO", originalEnv)

			os.Setenv("GITALY_CONNECTION_INFO", tt.envValue)

			config, err := readConfig(tt.repoPath, tt.projectPath)

			if tt.expectedError {
				require.Error(t, err)
				return
			}

			require.NoError(t, err)
			require.Equal(t, tt.expectedConfig, config)
		})
	}
}

func TestConfigureLogger(t *testing.T) {
	tests := []struct {
		name      string
		debugEnv  string
		expectErr bool
	}{
		{
			name:     "debug mode enabled",
			debugEnv: "1",
		},
		{
			name:     "debug mode disabled",
			debugEnv: "",
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			originalDebug := os.Getenv("DEBUG")
			defer func() {
				if originalDebug == "" {
					os.Unsetenv("DEBUG")
				} else {
					os.Setenv("DEBUG", originalDebug)
				}
			}()

			if tt.debugEnv != "" {
				os.Setenv("DEBUG", tt.debugEnv)
			} else {
				os.Unsetenv("DEBUG")
			}

			closer, err := configureLogger()

			if tt.expectErr {
				require.Error(t, err)
				return
			}

			require.NoError(t, err)
			require.NotNil(t, closer)
			closer.Close()
		})
	}
}

func TestGenerateCorrelationID(t *testing.T) {
	tests := []struct {
		name        string
		envValue    string
		expectEmpty bool
	}{
		{
			name:     "correlation ID from environment",
			envValue: "test-correlation-id-123",
		},
		{
			name:        "generate random ID when env is empty",
			envValue:    "",
			expectEmpty: false,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			originalEnv := os.Getenv(envCorrelationIDKey)
			defer func() {
				if originalEnv == "" {
					os.Unsetenv(envCorrelationIDKey)
				} else {
					os.Setenv(envCorrelationIDKey, originalEnv)
				}
			}()

			if tt.envValue != "" {
				os.Setenv(envCorrelationIDKey, tt.envValue)
			} else {
				os.Unsetenv(envCorrelationIDKey)
			}

			correlationID := generateCorrelationID()

			if tt.envValue != "" {
				require.Equal(t, tt.envValue, correlationID)
			} else {
				require.NotEmpty(t, correlationID)
			}
		})
	}
}

func TestGenerateProjectPermissions(t *testing.T) {
	tests := []struct {
		name                  string
		visibilityLevel       int
		repositoryAccessLevel int
		expectedPermissions   *indexer.ProjectPermissions
	}{
		{
			name:                  "valid permissions",
			visibilityLevel:       10,
			repositoryAccessLevel: 20,
			expectedPermissions: &indexer.ProjectPermissions{
				VisibilityLevel:       10,
				RepositoryAccessLevel: 20,
			},
		},
		{
			name:                  "missing visibility level",
			visibilityLevel:       -1,
			repositoryAccessLevel: 20,
			expectedPermissions:   nil,
		},
		{
			name:                  "missing repository access level",
			visibilityLevel:       10,
			repositoryAccessLevel: -1,
			expectedPermissions:   nil,
		},
		{
			name:                  "both missing",
			visibilityLevel:       -1,
			repositoryAccessLevel: -1,
			expectedPermissions:   nil,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			// Save original flag values
			originalVisibility := *visibilityLevelFlag
			originalRepository := *repositoryAccessLevelFlag

			// Set test values
			*visibilityLevelFlag = tt.visibilityLevel
			*repositoryAccessLevelFlag = tt.repositoryAccessLevel

			// Restore original values
			defer func() {
				*visibilityLevelFlag = originalVisibility
				*repositoryAccessLevelFlag = originalRepository
			}()

			permissions := generateProjectPermissions()
			require.Equal(t, tt.expectedPermissions, permissions)
		})
	}
}

func TestGenerateWikiPermissions(t *testing.T) {
	tests := []struct {
		name                string
		visibilityLevel     int
		wikiAccessLevel     int
		expectedPermissions *indexer.WikiPermissions
	}{
		{
			name:            "valid wiki permissions",
			visibilityLevel: 0,
			wikiAccessLevel: 10,
			expectedPermissions: &indexer.WikiPermissions{
				VisibilityLevel: 0,
				WikiAccessLevel: 10,
			},
		},
		{
			name:                "missing visibility level",
			visibilityLevel:     -1,
			wikiAccessLevel:     10,
			expectedPermissions: nil,
		},
		{
			name:                "missing wiki access level",
			visibilityLevel:     10,
			wikiAccessLevel:     -1,
			expectedPermissions: nil,
		},
		{
			name:                "both missing",
			visibilityLevel:     -1,
			wikiAccessLevel:     -1,
			expectedPermissions: nil,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			// Save original flag values
			originalVisibility := *visibilityLevelFlag
			originalWiki := *wikiAccessLevelFlag

			// Set test values
			*visibilityLevelFlag = tt.visibilityLevel
			*wikiAccessLevelFlag = tt.wikiAccessLevel

			// Restore original values
			defer func() {
				*visibilityLevelFlag = originalVisibility
				*wikiAccessLevelFlag = originalWiki
			}()

			permissions := generateWikiPermissions()
			require.Equal(t, tt.expectedPermissions, permissions)
		})
	}
}

func TestLoadConfig(t *testing.T) {
	tests := []struct {
		name                  string
		projectID             int64
		groupID               int64
		traversalIds          string
		hashedRootNamespaceId int16
		archived              string
		schemaVersionBlob     uint16
		schemaVersionCommit   uint16
		schemaVersionWiki     uint16
		visibilityLevel       int
		repositoryAccessLevel int
		wikiAccessLevel       int
		expectError           bool
	}{
		{
			name:                  "valid project config",
			projectID:             123,
			groupID:               0,
			traversalIds:          "1-2-3-",
			hashedRootNamespaceId: 456,
			archived:              "false",
			schemaVersionBlob:     2401,
			schemaVersionCommit:   2401,
			schemaVersionWiki:     2401,
			visibilityLevel:       10,
			repositoryAccessLevel: 20,
			wikiAccessLevel:       10,
		},
		{
			name:                  "valid group config",
			projectID:             0,
			groupID:               789,
			traversalIds:          "7-8-9-",
			hashedRootNamespaceId: 101,
			archived:              "true",
			schemaVersionBlob:     2402,
			schemaVersionCommit:   2402,
			schemaVersionWiki:     2402,
			visibilityLevel:       -1,
			repositoryAccessLevel: -1,
			wikiAccessLevel:       -1,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			// Set up environment variables that ConfigFromEnv might use
			originalEnvVars := map[string]string{
				"ELASTIC_CONNECTION_INFO": os.Getenv("ELASTIC_CONNECTION_INFO"),
				"RAILS_ENV":               os.Getenv("RAILS_ENV"),
			}
			defer func() {
				for key, val := range originalEnvVars {
					if val == "" {
						os.Unsetenv(key)
					} else {
						os.Setenv(key, val)
					}
				}
			}()

			// Set minimal required environment for elastic config
			os.Setenv("ELASTIC_CONNECTION_INFO", `{"url":["http://localhost:9200"],"index_name":"test-index"}`)
			os.Setenv("RAILS_ENV", "test")

			// Save and set flag values
			originalVisibility := *visibilityLevelFlag
			originalRepository := *repositoryAccessLevelFlag
			originalWiki := *wikiAccessLevelFlag

			*visibilityLevelFlag = tt.visibilityLevel
			*repositoryAccessLevelFlag = tt.repositoryAccessLevel
			*wikiAccessLevelFlag = tt.wikiAccessLevel

			defer func() {
				*visibilityLevelFlag = originalVisibility
				*repositoryAccessLevelFlag = originalRepository
				*wikiAccessLevelFlag = originalWiki
			}()

			config, err := loadConfig(
				tt.projectID,
				tt.groupID,
				tt.traversalIds,
				tt.hashedRootNamespaceId,
				tt.archived,
				tt.schemaVersionBlob,
				tt.schemaVersionCommit,
				tt.schemaVersionWiki,
			)

			if tt.expectError {
				require.Error(t, err)
				return
			}

			require.NoError(t, err)
			require.NotNil(t, config)
			require.Equal(t, tt.projectID, config.ProjectID)
			require.Equal(t, tt.groupID, config.GroupID)
			require.Equal(t, tt.traversalIds, config.TraversalIDs)
			require.Equal(t, tt.hashedRootNamespaceId, config.HashedRootNamespaceId)
			require.Equal(t, tt.archived, config.Archived)
			require.Equal(t, tt.schemaVersionBlob, config.SchemaVersionBlob)
			require.Equal(t, tt.schemaVersionCommit, config.SchemaVersionCommit)
			require.Equal(t, tt.schemaVersionWiki, config.SchemaVersionWiki)

			// Check permissions are set correctly
			if tt.visibilityLevel != -1 && tt.repositoryAccessLevel != -1 {
				require.NotNil(t, config.Permissions)
				require.Equal(t, int8(tt.visibilityLevel), config.Permissions.VisibilityLevel)
				require.Equal(t, int8(tt.repositoryAccessLevel), config.Permissions.RepositoryAccessLevel)
			} else {
				require.Nil(t, config.Permissions)
			}

			if tt.visibilityLevel != -1 && tt.wikiAccessLevel != -1 {
				require.NotNil(t, config.PermissionsWiki)
				require.Equal(t, int8(tt.visibilityLevel), config.PermissionsWiki.VisibilityLevel)
				require.Equal(t, int8(tt.wikiAccessLevel), config.PermissionsWiki.WikiAccessLevel)
			} else {
				require.Nil(t, config.PermissionsWiki)
			}
		})
	}
}
