package types

type AdapterType string

const (
	PostgreSQLAdapter    AdapterType = "postgresql"
	OpenSearchAdapter    AdapterType = "opensearch"
	ElasticsearchAdapter AdapterType = "elasticsearch"
)

var ValidAdapters = [...]AdapterType{
	PostgreSQLAdapter,
	OpenSearchAdapter,
	ElasticsearchAdapter,
}

type Connection interface {
	GetAdapterType() AdapterType
}

// PostgreSQLConnection holds connection details for PostgreSQL
type PostgreSQLConnection struct {
	Host     string `json:"host"`
	Port     uint16 `json:"port"`
	User     string `json:"user"`
	Password string `json:"password"`
	Database string `json:"database"`
	Table    string `json:"table"`
}

// OpenSearchConnection holds connection details for OpenSearch
type OpenSearchConnection struct {
	URL            []string `json:"url"`
	AWS            bool     `json:"aws"`
	Region         string   `json:"aws_region"`
	AccessKey      string   `json:"aws_access_key"`
	SecretKey      string   `json:"aws_secret_access_key"`
	RoleARN        string   `json:"aws_role_arn"`
	RequestTimeout int      `json:"client_request_timeout"`
}

// ElasticsearchConnection holds connection details for Elasticsearch
type ElasticsearchConnection struct {
	URL []string `json:"url"`
}

func (c PostgreSQLConnection) GetAdapterType() AdapterType {
	return PostgreSQLAdapter
}

func (c OpenSearchConnection) GetAdapterType() AdapterType {
	return OpenSearchAdapter
}

func (c ElasticsearchConnection) GetAdapterType() AdapterType {
	return ElasticsearchAdapter
}
