package streamer

import (
	"bytes"
	"testing"

	"github.com/stretchr/testify/require"
)

func TestNew(t *testing.T) {
	t.Parallel()

	writer := &bytes.Buffer{}
	s := New(writer, "")

	require.NotNil(t, s)
	require.Equal(t, writer, s.writer)
	require.NotNil(t, s.csvWriter)
	require.Empty(t, s.headerSent)
	require.Equal(t, DefaultSectionSeparator, s.sectionSeparator)
}

func TestNewWithCustomSeparator(t *testing.T) {
	t.Parallel()

	writer := &bytes.Buffer{}
	customSeparator := "===SECTION==="
	s := New(writer, customSeparator)

	require.NotNil(t, s)
	require.Equal(t, writer, s.writer)
	require.NotNil(t, s.csvWriter)
	require.Empty(t, s.headerSent)
	require.Equal(t, customSeparator, s.sectionSeparator)
}

func TestNewStdout(t *testing.T) {
	t.Parallel()

	s := NewStdout()
	require.NotNil(t, s)
	require.NotNil(t, s.writer)
	require.NotNil(t, s.csvWriter)
	require.Equal(t, DefaultSectionSeparator, s.sectionSeparator)
}

func TestStreamSingle(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name    string
		record  Record
		wantOut string
		wantErr bool
	}{
		{
			name: "version info record",
			record: &IndexerVersionInfo{
				Version:   "1.0.0",
				BuildTime: "2023-01-01",
			},
			wantOut: "--section-start--\nversion,build_time\n1.0.0,2023-01-01\n",
			wantErr: false,
		},
		{
			name: "chunk info record",
			record: &IndexedChunkInfo{
				ID: "chunk-123",
			},
			wantOut: "--section-start--\nid\nchunk-123\n",
			wantErr: false,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			buf := &bytes.Buffer{}
			s := New(buf, "")

			err := s.StreamSingle(tt.record)

			if tt.wantErr {
				require.Error(t, err)
				return
			}

			require.NoError(t, err)
			require.Equal(t, tt.wantOut, buf.String())
		})
	}
}

func TestStream(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name     string
		records  []Record
		wantOut  string
		wantErr  bool
		errCheck func(t *testing.T, err error)
	}{
		{
			name:    "empty records",
			records: []Record{},
			wantOut: "",
			wantErr: false,
		},
		{
			name: "single version info record",
			records: []Record{
				&IndexerVersionInfo{
					Version:   "1.0.0",
					BuildTime: "2023-01-01",
				},
			},
			wantOut: "--section-start--\nversion,build_time\n1.0.0,2023-01-01\n",
			wantErr: false,
		},
		{
			name: "multiple version info records",
			records: []Record{
				&IndexerVersionInfo{
					Version:   "1.0.0",
					BuildTime: "2023-01-01",
				},
				&IndexerVersionInfo{
					Version:   "1.1.0",
					BuildTime: "2023-02-01",
				},
			},
			wantOut: "--section-start--\nversion,build_time\n1.0.0,2023-01-01\n1.1.0,2023-02-01\n",
			wantErr: false,
		},
		{
			name: "mixed record types",
			records: []Record{
				&IndexerVersionInfo{
					Version:   "1.0.0",
					BuildTime: "2023-01-01",
				},
				&IndexedChunkInfo{
					ID: "chunk-123",
				},
			},
			wantOut: "",
			wantErr: true,
			errCheck: func(t *testing.T, err error) {
				require.Contains(t, err.Error(), "record at index 1 has type indexed_chunk_info, expected indexer_version_info")
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			buf := &bytes.Buffer{}
			s := New(buf, "")

			err := s.Stream(tt.records)

			if tt.wantErr {
				require.Error(t, err)
				if tt.errCheck != nil {
					tt.errCheck(t, err)
				}
				return
			}

			require.NoError(t, err)
			require.Equal(t, tt.wantOut, buf.String())
		})
	}
}

func TestStreamerMultipleTypes(t *testing.T) {
	t.Parallel()

	buf := &bytes.Buffer{}
	s := New(buf, "")

	// Stream first record type
	err := s.StreamSingle(&IndexerVersionInfo{
		Version:   "1.0.0",
		BuildTime: "2023-01-01",
	})
	require.NoError(t, err)

	// Stream second record type
	err = s.StreamSingle(&IndexedChunkInfo{
		ID: "chunk-123",
	})
	require.NoError(t, err)

	// Stream first record type again
	err = s.StreamSingle(&IndexerVersionInfo{
		Version:   "1.1.0",
		BuildTime: "2023-02-01",
	})
	require.NoError(t, err)

	// Expected output: headers are resent when switching between record types
	// with section separators
	expected := "--section-start--\n" +
		"version,build_time\n" +
		"1.0.0,2023-01-01\n" +
		"--section-start--\n" +
		"id\n" +
		"chunk-123\n" +
		"--section-start--\n" +
		"version,build_time\n" +
		"1.1.0,2023-02-01\n"

	require.Equal(t, expected, buf.String())
}

func TestStreamerWithCustomSeparator(t *testing.T) {
	t.Parallel()

	buf := &bytes.Buffer{}
	customSeparator := "===SECTION==="
	s := New(buf, customSeparator)

	// Stream first record type
	err := s.StreamSingle(&IndexerVersionInfo{
		Version:   "1.0.0",
		BuildTime: "2023-01-01",
	})
	require.NoError(t, err)

	// Stream second record type
	err = s.StreamSingle(&IndexedChunkInfo{
		ID: "chunk-123",
	})
	require.NoError(t, err)

	// Expected output with custom separator
	expected := "===SECTION===\n" +
		"version,build_time\n" +
		"1.0.0,2023-01-01\n" +
		"===SECTION===\n" +
		"id\n" +
		"chunk-123\n"

	require.Equal(t, expected, buf.String())
}

func TestStreamerClose(t *testing.T) {
	t.Parallel()

	// Test with non-closer writer
	t.Run("non-closer writer", func(t *testing.T) {
		t.Parallel()

		buf := &bytes.Buffer{}
		s := New(buf, "")

		err := s.Close()
		require.NoError(t, err)
	})

	// Test with mock closer
	t.Run("closing writer", func(t *testing.T) {
		t.Parallel()

		writer := &mockCloser{
			buf:       &bytes.Buffer{},
			closeFunc: func() error { return nil },
		}
		s := New(writer, "")

		err := s.Close()
		require.NoError(t, err)
		require.True(t, writer.closed)
	})
}

// mockCloser implements io.Writer and io.Closer for testing
type mockCloser struct {
	buf       *bytes.Buffer
	closed    bool
	closeFunc func() error
}

func (m *mockCloser) Write(p []byte) (n int, err error) {
	return m.buf.Write(p)
}

func (m *mockCloser) Close() error {
	m.closed = true
	return m.closeFunc()
}
