package elasticsearch

import (
	"context"
	"fmt"
	"log/slog"

	"github.com/olivere/elastic/v7"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/types"
)

type ElasticsearchClient struct {
	Conn   types.ElasticsearchConnection
	Client *elastic.Client
}

func New(conn types.ElasticsearchConnection) *ElasticsearchClient {
	return &ElasticsearchClient{
		Conn: conn,
	}
}

func (c *ElasticsearchClient) Connect(ctx context.Context) error {
	slog.Debug("connecting to Elasticsearch", "urls", c.Conn.URL)

	if len(c.Conn.URL) == 0 {
		return fmt.Errorf("failed to create Elasticsearch client: no URLs provided")
	}

	opts := []elastic.ClientOptionFunc{
		elastic.SetURL(c.Conn.URL...),
		elastic.SetSniff(false),
		elastic.SetHealthcheck(false),
	}

	client, err := elastic.NewClient(opts...)
	if err != nil {
		return fmt.Errorf("failed to create Elasticsearch client: %w", err)
	}

	// Temporary: call ping to verify the connection
	info, code, err := client.Ping(c.Conn.URL[0]).Do(ctx)
	if err != nil {
		return fmt.Errorf("failed to ping Elasticsearch: %w", err)
	}

	slog.Info("Elasticsearch connection successful",
		"status", code,
		"version", info.Version.Number,
		"cluster_name", info.Name,
	)

	c.Client = client
	return nil
}

func (c *ElasticsearchClient) Close() error {
	if c.Client != nil {
		c.Client.Stop()
	}
	return nil
}
