package size

import (
	"context"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/chunker"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/types"
)

const simpleChunkerTestContent = `This is a test content for chunking.
We will split this content into multiple chunks of a certain size.
We will use the simple size chunker.

This is not meant for testing with a parser,
since the parser is specifically for code content.

The simple chunker should split this content given a chunk size.
If the split does not occur at a new line, the simple chunker
will try to split at the last new line index.
`

const simpleChunkerTestChunk1 = `This is a test content for chunking.
We will split this content into multiple chunks of a certain size.
We will use the simple size chunker.

This is not meant for testing with a parser,
`

const simpleChunkerTestChunk2 = `since the parser is specifically for code content.

The simple chunker should split this content given a chunk size.
If the split does not occur at a new line, the simple chunker
`

const simpleChunkerTestChunk3 = `will try to split at the last new line index.
`

// Tests for New function

func TestNew(t *testing.T) {
	t.Parallel()

	options := chunker.ChunkOptions{
		ChunkSize:    200,
		ChunkOverlap: 0,
	}

	// Test that the size chunker correctly implements the Chunker interface
	sizeChunker, err := New(options)

	require.NoError(t, err)
	require.Implements(t, (*chunker.Chunker)(nil), sizeChunker)
}

func TestNewWithZeroChunkSize(t *testing.T) {
	t.Parallel()

	options := chunker.ChunkOptions{
		ChunkSize: 0,
	}

	files := []types.File{
		{
			Path:    "file1.txt",
			Content: "This is a short file",
		},
	}

	// Test that this does not result in an infinite loop,
	// since the chunk size will be overridden with the default chunk size value

	sizeChunker, err := New(options)
	require.NoError(t, err)

	chunks, err := sizeChunker.ChunkFiles(context.TODO(), files)
	require.NoError(t, err)
	require.Len(t, chunks, 1)
}

func TestNewWithTooLargeChunkOverlap(t *testing.T) {
	t.Parallel()

	options := chunker.ChunkOptions{
		ChunkSize:    10,
		ChunkOverlap: 10,
	}

	sizeChunker, err := New(options)
	require.Nil(t, sizeChunker)
	require.Error(t, err)
	require.Contains(t, err.Error(), "chunk overlap (10) must be less than chunk size (10)")
}

// Tests for ChunkFiles function

func TestChunkFiles(t *testing.T) {
	t.Parallel()

	options := chunker.ChunkOptions{
		ChunkSize: 200,
	}

	files := []types.File{
		{
			Path:    "dir/file1.txt",
			Content: "This is a short file",
			OID:     "abc123",
		},
		{
			Path:    "dir/file2.txt",
			Content: simpleChunkerTestContent,
			OID:     "xyz789",
		},
	}

	sizeChunker, _ := New(options)
	chunks, err := sizeChunker.ChunkFiles(context.TODO(), files)
	require.NoError(t, err)

	// First file should generate 1 chunk, second file should generate 3 chunks
	require.Len(t, chunks, 4)

	// Check first file's chunk
	file1chunk := chunks[0]
	require.Equal(t, "dir/file1.txt", file1chunk.Path)
	require.Equal(t, chunker.FileContentType, file1chunk.Type)
	require.Equal(t, "file1.txt", file1chunk.Name)
	require.Empty(t, file1chunk.Language)
	require.Equal(t, "This is a short file", file1chunk.Content)
	require.Equal(t, "abc123", file1chunk.OID)
	require.Equal(t, 0, file1chunk.StartByte)
	require.Equal(t, len(files[0].Content), file1chunk.Length)

	// Check that the second file's chunks are correct
	for _, chunk := range chunks[1:3] {
		require.Equal(t, "dir/file2.txt", chunk.Path)
		require.Equal(t, chunker.FileContentType, chunk.Type)
		require.Equal(t, "file2.txt", chunk.Name)
		require.Empty(t, chunk.Language)
		require.Equal(t, "xyz789", chunk.OID)
	}

	file2chunk1 := chunks[1]
	expectedChunk1StartByte := 0
	expectedChunk1Length := len(simpleChunkerTestChunk1)
	require.Equal(t, simpleChunkerTestChunk1, file2chunk1.Content)
	require.Equal(t, expectedChunk1StartByte, file2chunk1.StartByte)
	require.Equal(t, expectedChunk1Length, file2chunk1.Length)

	file2chunk2 := chunks[2]
	expectedChunk2StartByte := len(simpleChunkerTestChunk1)
	expectedChunk2Length := len(simpleChunkerTestChunk2)
	require.Equal(t, simpleChunkerTestChunk2, file2chunk2.Content)
	require.Equal(t, expectedChunk2StartByte, file2chunk2.StartByte)
	require.Equal(t, expectedChunk2Length, file2chunk2.Length)

	file2chunk3 := chunks[3]
	expectedChunk3StartByte := len(simpleChunkerTestChunk1) + len(simpleChunkerTestChunk2)
	expectedChunk3Length := len(simpleChunkerTestChunk3)
	require.Equal(t, simpleChunkerTestChunk3, file2chunk3.Content)
	require.Equal(t, expectedChunk3StartByte, file2chunk3.StartByte)
	require.Equal(t, expectedChunk3Length, file2chunk3.Length)
}

func TestChunkFilesWithOverlap(t *testing.T) {
	t.Parallel()

	options := chunker.ChunkOptions{
		ChunkSize:    5,
		ChunkOverlap: 1,
	}

	files := []types.File{
		{
			Path:    "test-file.txt",
			Content: "1234567890",
			OID:     "abc123",
		},
	}

	sizeChunker, _ := New(options)
	chunks, err := sizeChunker.ChunkFiles(context.TODO(), files)
	require.NoError(t, err)
	require.Len(t, chunks, 3)

	chunk1 := chunks[0]
	require.Equal(t, "12345", chunk1.Content)
	require.Equal(t, 0, chunk1.StartByte)

	chunk2 := chunks[1]
	require.Equal(t, "56789", chunk2.Content)
	require.Equal(t, 4, chunk2.StartByte)

	chunk3 := chunks[2]
	require.Equal(t, "90", chunk3.Content)
	require.Equal(t, 8, chunk3.StartByte)
}

func TestChunkFilesEmpty(t *testing.T) {
	t.Parallel()

	options := chunker.ChunkOptions{
		ChunkSize: 200,
	}

	sizeChunker, _ := New(options)
	chunks, err := sizeChunker.ChunkFiles(context.TODO(), []types.File{})
	require.NoError(t, err)
	require.Empty(t, chunks)
}

func TestChunkFilesWithTimeoutContext(t *testing.T) {
	t.Parallel()

	files := []types.File{
		{
			Path:    "file1.txt",
			Content: "This is a short file",
			OID:     "abc123",
		},
	}

	options := chunker.ChunkOptions{
		ChunkSize: 200,
	}
	sizeChunker, _ := New(options)

	ctx, cancel := context.WithTimeout(context.Background(), 1*time.Microsecond)
	defer cancel()

	time.Sleep(1 * time.Millisecond)

	// At this point, the context already timed out,
	// so we are expecting this to return an error
	chunks, err := sizeChunker.ChunkFiles(ctx, files)
	require.Error(t, err)
	require.Nil(t, chunks)
}
