package code

import (
	"context"
	"fmt"

	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/chunker"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/types"

	codechunker "gitlab.com/gitlab-org/rust/gitlab-code-parser/bindings/go/chunker"
)

// Chunk information related to the file
type ChunkFileInfo struct {
	Name string
	OID  string
}

// Chunker implements the chunker.Chunker interface using code-parser chunking
type Chunker struct {
	chunker *codechunker.Chunker
}

// New creates a new code chunker with the provided options
func New(options chunker.ChunkOptions) (*Chunker, error) {
	chunkSize := options.ChunkSize
	if chunkSize <= 0 {
		chunkSize = chunker.DefaultChunkSize
	}

	// If chunk overlap is greater than chunk size, return an error
	if options.ChunkOverlap >= chunkSize {
		return nil, fmt.Errorf("chunk overlap (%d) must be less than chunk size (%d)", options.ChunkOverlap, chunkSize)
	}

	// Initialize the split-code chunker
	// This chunker falls back to a size chunker if the language is not supported
	codeChunker, err := codechunker.NewChunkerSplitCode(int(chunkSize))
	if err != nil {
		return nil, fmt.Errorf("code_byte chunker: %w", err)
	}

	return &Chunker{
		chunker: codeChunker,
	}, nil
}

// NewPreBert creates a new code chunker with pre-bert token size limits.
func NewPreBert(options chunker.ChunkOptions) (*Chunker, error) {
	chunkSize := options.ChunkSize
	if chunkSize <= 0 {
		chunkSize = chunker.DefaultChunkSize
	}

	// If chunk overlap is greater than chunk size, return an error
	if options.ChunkOverlap >= chunkSize {
		return nil, fmt.Errorf("chunk overlap (%d) must be less than chunk size (%d)", options.ChunkOverlap, chunkSize)
	}

	codeChunker, err := codechunker.NewChunkerSplitCodePreBert(int(chunkSize))
	if err != nil {
		return nil, fmt.Errorf("code_pre_bert chunker: %w", err)
	}

	return &Chunker{
		chunker: codeChunker,
	}, nil
}

// ChunkFiles implements the chunker.Chunker interface
func (c *Chunker) ChunkFiles(ctx context.Context, files []types.File) ([]chunker.Chunk, error) {
	fileInfoMap := make(map[string]ChunkFileInfo)
	for _, file := range files {
		fileInfoMap[file.Path] = ChunkFileInfo{
			Name: file.GetFilename(),
			OID:  file.OID,
		}
		c.chunker.AddFile(file.Path, file.Content)
	}

	if err := c.chunker.ChunkFiles(); err != nil {
		return nil, fmt.Errorf("codeChunker.ChunkFiles: %w", err)
	}

	chunks := make([]chunker.Chunk, 0)
	for codeChunk := range c.chunker.Chunks() {
		chunks = append(chunks, chunker.Chunk{
			Path:      codeChunk.FilePath,
			Type:      chunker.FileContentType,
			Name:      fileInfoMap[codeChunk.FilePath].Name,
			Language:  codeChunk.Language,
			Content:   codeChunk.Content,
			OID:       fileInfoMap[codeChunk.FilePath].OID,
			StartByte: codeChunk.StartByte,
			Length:    codeChunk.Length,
			StartLine: codeChunk.StartLine,
		})
	}

	c.chunker.Clear()

	return chunks, nil
}

func (c *Chunker) Close() {
	c.chunker.Close()
}
