package chunker_test

import (
	"context"
	"fmt"
	"os"
	"testing"

	"github.com/stretchr/testify/require"
	. "gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/chunker"
	codeChunker "gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/chunker/code"
	sizeChunker "gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/chunker/size"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/chunk/types"
)

func BenchmarkChunker(b *testing.B) {
	for _, tc := range []struct {
		desc    string
		chunker func(tb testing.TB) Chunker
	}{
		{
			desc: "size",
			chunker: func(tb testing.TB) Chunker {
				c, err := sizeChunker.New(ChunkOptions{})
				require.NoError(tb, err)

				return c
			},
		},
		{
			desc: "code_bytes",
			chunker: func(tb testing.TB) Chunker {
				c, err := codeChunker.New(ChunkOptions{})
				require.NoError(tb, err)

				return c
			},
		},
		{
			desc: "code_pre_bert",
			chunker: func(tb testing.TB) Chunker {
				c, err := codeChunker.NewPreBert(ChunkOptions{})
				require.NoError(tb, err)

				return c
			},
		},
	} {
		b.Run(tc.desc, func(b *testing.B) {
			content, err := os.ReadFile("../chunk.go")
			require.NoError(b, err)

			var files []types.File
			for i := range 10 {
				files = append(files, types.File{
					Path:    fmt.Sprintf("chunk_%d.go", i),
					Content: string(content),
				})
			}

			c := tc.chunker(b)
			ctx, cancel := context.WithCancel(context.Background())
			b.Cleanup(cancel)

			b.ResetTimer()

			for range b.N {
				chunks, err := c.ChunkFiles(ctx, files)
				require.NoError(b, err)
				require.GreaterOrEqual(b, len(chunks), len(files), "expected at least as many chunks as input files")
			}
		})
	}
}
