package elastic_test

import (
	"context"
	"crypto/tls"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"os"
	"regexp"
	"strings"
	"sync"
	"testing"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/defaults"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/gitlab-elasticsearch-indexer/internal/mode/advanced/elastic"
)

const (
	projectID       = int64(667)
	projectIDString = "667"
)

const credsRespTmpl = `{
  "Code": "Success",
  "Type": "AWS-HMAC",
  "AccessKeyId" : "accessKey",
  "SecretAccessKey" : "secret",
  "Token" : "token",
  "Expiration" : "%s",
  "LastUpdated" : "2009-11-23T0:00:00Z"
}`

const credsFailRespTmpl = `{
  "Code": "ErrorCode",
  "Message": "ErrorMsg",
  "LastUpdated": "2009-11-23T0:00:00Z"
}`

type testResolver struct {
	endpoint string
}

func (tr testResolver) EndpointFor(service, region string, opts ...func(*endpoints.Options)) (endpoints.ResolvedEndpoint, error) {
	return endpoints.ResolvedEndpoint{URL: tr.endpoint}, nil
}

func initTestServer(expireOn string, failAssume bool, provideToken bool) *httptest.Server { //nolint:unparam
	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		switch r.URL.Path {
		case "/latest/meta-data/iam/security-credentials/":
			fmt.Fprintln(w, "RoleName")
		case "/latest/meta-data/iam/security-credentials/RoleName":
			if failAssume {
				fmt.Fprintf(w, "%s", credsFailRespTmpl)
			} else {
				fmt.Fprintf(w, credsRespTmpl, expireOn)
			}
		case "/gitlab-index-test/_doc/667":
			time.Sleep(3 * time.Second)
			fmt.Fprintln(w, "{}")
		case "/latest/api/token":
			if provideToken {
				fmt.Fprintf(w, "test-token")
			} else {
				http.Error(w, "Not Found", http.StatusNotFound)
			}
		default:
			http.Error(w, "bad request", http.StatusBadRequest)
		}
	}))

	return server
}

func TestResolveAWSCredentialsStatic(t *testing.T) {
	awsConfig := &aws.Config{}
	config, err := elastic.ReadConfig(strings.NewReader(
		`{
			"url":["http://localhost:9200"],
			"aws":true,
			"aws_access_key": "static_access_key",
			"aws_secret_access_key": "static_secret_access_key"
		}`,
	))
	require.NoError(t, err)

	creds := elastic.ResolveAWSCredentials(config, awsConfig)
	credsValue, err := creds.Get()
	require.NoError(t, err)
	require.Equal(t, "static_access_key", credsValue.AccessKeyID, "Expect access key ID to match")
	require.Equal(t, "static_secret_access_key", credsValue.SecretAccessKey, "Expect secret access key to match")
}

func TestResolveAWSEnvCredentials(t *testing.T) {
	awsConfig := &aws.Config{}
	config, err := elastic.ReadConfig(strings.NewReader(
		`{
			"url":["http://localhost:9200"],
			"aws":true
		}`,
	))
	require.NoError(t, err)

	os.Setenv("AWS_ACCESS_KEY_ID", "id")
	os.Setenv("AWS_SECRET_ACCESS_KEY", "secret")
	os.Setenv("AWS_SESSION_TOKEN", "session-token")
	defer func() {
		os.Unsetenv("AWS_ACCESS_KEY_ID")
		os.Unsetenv("AWS_SECRET_ACCESS_KEY")
		os.Unsetenv("AWS_SESSION_TOKEN")
	}()

	creds := elastic.ResolveAWSCredentials(config, awsConfig)
	credsValue, err := creds.Get()
	require.NoError(t, err)
	require.Equal(t, "id", credsValue.AccessKeyID, "Expect access key ID to match")
	require.Equal(t, "secret", credsValue.SecretAccessKey, "Expect secret access key to match")
	require.Equal(t, "session-token", credsValue.SessionToken, "Expect session token to match")
}

func TestResolveAWSCredentialsEc2RoleProfiles(t *testing.T) {
	servers := [...]*httptest.Server{
		initTestServer("2014-12-16T01:51:37Z", false, false),
		initTestServer("2014-12-16T01:51:37Z", false, true),
	}

	for _, server := range servers {
		defer server.Close()

		awsConfig := defaults.Config().WithEndpointResolver(testResolver{endpoint: server.URL + "/latest"})

		config, err := elastic.ReadConfig(strings.NewReader(
			`{
				"url":["` + server.URL + `"],
				"aws":true,
				"aws_region":"us-east-1",
				"aws_profile":"test_aws_will_not_find"
			}`,
		))
		require.NoError(t, err)

		os.Setenv("AWS_SHARED_CREDENTIALS_FILE", "/tmp/notexist")
		defer os.Unsetenv("AWS_SHARED_CREDENTIALS_FILE")

		creds := elastic.ResolveAWSCredentials(config, awsConfig)
		credsValue, err := creds.Get()
		require.NoError(t, err)
		require.Equal(t, "accessKey", credsValue.AccessKeyID, "Expect access key ID to match")
		require.Equal(t, "secret", credsValue.SecretAccessKey, "Expect secret access key to match")
	}
}

func TestResolveAWSCredentialsECSCredsProvider(t *testing.T) {
	server := initTestServer("2014-12-16T01:51:37Z", false, false)
	defer server.Close()

	awsConfig := &aws.Config{
		HTTPClient: &http.Client{},
	}

	config, err := elastic.ReadConfig(strings.NewReader(
		`{
			"url":["` + server.URL + `"],
			"aws":true,
			"aws_region":"us-east-1",
			"aws_profile":"test_aws_will_not_find"
		}`,
	))
	require.NoError(t, err)

	os.Setenv("AWS_SHARED_CREDENTIALS_FILE", "/tmp/notexist")
	os.Setenv("AWS_CONTAINER_CREDENTIALS_FULL_URI", server.URL+"/latest/meta-data/iam/security-credentials/RoleName")
	defer func() {
		defer os.Unsetenv("AWS_SHARED_CREDENTIALS_FILE")
		defer os.Unsetenv("AWS_CONTAINER_CREDENTIALS_FULL_URI")
	}()

	creds := elastic.ResolveAWSCredentials(config, awsConfig)
	credsValue, err := creds.Get()
	require.NoError(t, err)
	require.Equal(t, "accessKey", credsValue.AccessKeyID, "Expect access key ID to match")
	require.Equal(t, "secret", credsValue.SecretAccessKey, "Expect secret access key to match")
}

func TestAWSConfiguration(t *testing.T) {
	var req *http.Request

	// httptest certificate is unsigned
	transport := http.DefaultTransport
	defer func() { http.DefaultTransport = transport }()
	http.DefaultTransport = &http.Transport{TLSClientConfig: &tls.Config{InsecureSkipVerify: true}}

	f := func(w http.ResponseWriter, r *http.Request) {
		req = r

		w.Header().Set("Content-Type", "application/json")
		_, err := w.Write([]byte(`{}`))
		if err != nil {
			t.Logf("Error writing response: %v", err)
			http.Error(w, "Internal server error", http.StatusInternalServerError)
			return
		}
	}

	srv := httptest.NewTLSServer(http.HandlerFunc(f))
	defer srv.Close()

	config, err := elastic.ReadConfig(strings.NewReader(
		`{
			"url":["` + srv.URL + `"],
			"aws":true,
			"aws_region": "us-east-1",
			"aws_access_key": "0",
			"aws_secret_access_key": "0"
		}`,
	))
	require.NoError(t, err)
	config.ProjectID = 633

	client, err := elastic.NewClient(config, "the-correlation-id")
	require.NoError(t, err)
	// intiate a ClusterHealth API call to Elasticsearch since SetHealthcheck is set to false
	_, err = client.Client.ClusterHealth().Do(context.Background())
	require.NoError(t, err)
	defer client.Close()

	require.NotNil(t, req)
	authRE := regexp.MustCompile(`\AAWS4-HMAC-SHA256 Credential=0/\d{8}/us-east-1/es/aws4_request, SignedHeaders=accept;content-type;date;host;x-amz-date;x-opaque-id, Signature=[a-f0-9]{64}\z`)
	require.Regexp(t, authRE, req.Header.Get("Authorization"))
	require.NotEmpty(t, req.Header.Get("X-Amz-Date"))
}

func setupTestClient(t *testing.T) *elastic.Client {
	if os.Getenv("ELASTIC_CONNECTION_INFO") == "" {
		t.Log("ELASTIC_CONNECTION_INFO not set")
		t.SkipNow()
	}

	os.Setenv("RAILS_ENV", fmt.Sprintf("test-elastic-%d", time.Now().Unix()))

	config, err := elastic.ConfigFromEnv()
	require.NoError(t, err)
	config.ProjectID = projectID
	config.IndexNameCommits = config.IndexNameDefault + "-commits"

	client, err := elastic.NewClient(config, "test-correlation-id")
	require.NoError(t, err)

	require.Equal(t, projectID, client.ParentID())

	return client
}

func setupTestClientAndCreateIndex(t *testing.T) *elastic.Client {
	client := setupTestClient(t)

	_ = client.DeleteIndex(client.IndexNameDefault)
	_ = client.DeleteIndex(client.IndexNameCommits)

	require.NoError(t, client.CreateDefaultWorkingIndex())
	require.NoError(t, client.CreateCommitsWorkingIndex())

	return client
}

func TestElasticClientIndexAndRetrieval(t *testing.T) {
	client := setupTestClientAndCreateIndex(t)

	blobDoc := map[string]interface{}{}
	client.Index("blob", projectIDString+"_foo", blobDoc)

	commitDoc := map[string]interface{}{}
	client.Index("commit", projectIDString+"_0000", commitDoc)

	require.NoError(t, client.Flush())

	blob, err := client.GetBlob("foo")
	require.NoError(t, err)
	require.True(t, blob.Found)

	commit, err := client.GetCommit("0000")
	require.NoError(t, err)
	require.True(t, commit.Found)

	client.Remove("blob", projectIDString+"_foo")
	require.NoError(t, client.Flush())

	_, err = client.GetBlob("foo")
	require.Error(t, err)

	// indexing a doc with unexpected field will cause an ES strict_dynamic_mapping_exception
	// for our IndexMapping
	blobDocInvalid := map[string]interface{}{fmt.Sprintf("invalid-key-%d", time.Now().Unix()): ""}
	client.Index("blob", projectIDString+"_invalid", blobDocInvalid)
	require.Error(t, client.Flush())

	require.NoError(t, client.DeleteIndex(client.IndexNameDefault))
	require.NoError(t, client.DeleteIndex(client.IndexNameCommits))
}

func TestFlushErrorWithESActionRequestValidationException(t *testing.T) {
	client := setupTestClient(t)

	// set IndexName empty here to simulate ES action_request_validation_exception,
	// so that the `err` param passed to `afterFunc` is not nil
	client.IndexNameDefault = ""
	blobDoc := map[string]interface{}{}
	client.Index("blob", projectIDString+"_foo", blobDoc)

	require.Error(t, client.Flush())
}

func TestElasticReadConfig(t *testing.T) {
	config, err := elastic.ReadConfig(strings.NewReader(
		`{
			"url":["http://elasticsearch:9200"],
			"index_name": "foobar"
		}`,
	))
	require.NoError(t, err)

	require.Equal(t, "foobar", config.IndexNameDefault)
	require.Equal(t, []string{"http://elasticsearch:9200"}, config.URL)
	require.Equal(t, elastic.DefaultMaxBulkSize, config.MaxBulkSize)
	require.Equal(t, elastic.DefaultBulkWorkers, config.BulkWorkers)
}

func TestElasticReadConfigCustomBulkSettings(t *testing.T) {
	config, err := elastic.ReadConfig(strings.NewReader(
		`{
			"max_bulk_size_bytes": 1024,
			"max_bulk_concurrency": 6
		}`,
	))
	require.NoError(t, err)

	require.Equal(t, 1024, config.MaxBulkSize)
	require.Equal(t, 6, config.BulkWorkers)

}

func TestCorrelationIdForwardedAsXOpaqueId(t *testing.T) {
	var req *http.Request

	f := func(w http.ResponseWriter, r *http.Request) {
		req = r

		w.Header().Set("Content-Type", "application/json")
		_, err := w.Write([]byte(`{}`))
		if err != nil {
			t.Logf("Error writing response: %v", err)
			http.Error(w, "Internal server error", http.StatusInternalServerError)
			return
		}
	}

	srv := httptest.NewServer(http.HandlerFunc(f))
	defer srv.Close()

	config, err := elastic.ReadConfig(strings.NewReader(
		`{
			"url":["` + srv.URL + `"]
		}`,
	))
	require.NoError(t, err)
	config.ProjectID = projectID

	client, err := elastic.NewClient(config, "the-correlation-id")
	require.NoError(t, err)

	blobDoc := map[string]interface{}{}
	client.Index("blob", projectIDString+"_foo", blobDoc)
	require.NoError(t, client.Flush())

	require.NotNil(t, req)
	require.Equal(t, "the-correlation-id", req.Header.Get("X-Opaque-Id"))
}

func TestClientTimeout(t *testing.T) {
	server := initTestServer("2014-12-16T01:51:37Z", false, false)
	defer server.Close()

	config := os.Getenv("ELASTIC_CONNECTION_INFO")
	os.Setenv(
		"ELASTIC_CONNECTION_INFO",
		`{
			"url": ["`+server.URL+`"],
			"index_name": "gitlab-index-test",
			"client_request_timeout": 1,
			"index_name_commits": "foobar-commits"
		}`,
	)
	defer os.Setenv("ELASTIC_CONNECTION_INFO", config)

	client := setupTestClient(t)
	require.NotNil(t, client)

	_, err := client.Get("blob", projectIDString)
	require.Error(t,
		err,
		"context deadline exceeded (Client.Timeout exceeded while awaiting headers)",
	)

	os.Setenv(
		"ELASTIC_CONNECTION_INFO",
		`{
			"url": ["`+server.URL+`"],
			"index_name": "gitlab-index-test",
			"index_name_commits": "foobar-commits"
		}`,
	)

	client = setupTestClient(t)
	require.NotNil(t, client)

	_, err = client.Get("blob", projectIDString)
	require.NoError(t, err)
}

func TestHealthcheckIsDisabled(t *testing.T) {
	var req *http.Request

	f := func(w http.ResponseWriter, r *http.Request) {
		req = r
	}

	srv := httptest.NewServer(http.HandlerFunc(f))
	defer srv.Close()

	client := setupTestClient(t)
	require.NotNil(t, client)
	defer client.Close()

	require.Nil(t, req)
}

func TestBulkSizeTracking(t *testing.T) {
	originalConfig := os.Getenv("ELASTIC_CONNECTION_INFO")
	defer os.Setenv("ELASTIC_CONNECTION_INFO", originalConfig)

	var configMap map[string]interface{}
	if originalConfig != "" {
		err := json.Unmarshal([]byte(originalConfig), &configMap)
		require.NoError(t, err)
	} else {
		configMap = make(map[string]interface{})
	}

	configMap["max_bulk_size_bytes"] = 1000
	modifiedConfig, err := json.Marshal(configMap)
	require.NoError(t, err)

	os.Setenv("ELASTIC_CONNECTION_INFO", string(modifiedConfig))

	client := setupTestClient(t)
	require.NotNil(t, client)
	defer client.Close()

	require.Equal(t, 0, client.CurrentBatchSize())

	smallDoc := map[string]interface{}{"field": "value", "field2": "value2"}

	initialSize := client.CurrentBatchSize()
	client.Index("blob", projectIDString+"_0000", smallDoc)
	require.Greater(t, client.CurrentBatchSize(), initialSize)

	err = client.Flush()
	require.NoError(t, err)
	require.Equal(t, 0, client.CurrentBatchSize())
}

func TestProactiveFlushOnSizeLimit(t *testing.T) {
	originalConfig := os.Getenv("ELASTIC_CONNECTION_INFO")
	defer os.Setenv("ELASTIC_CONNECTION_INFO", originalConfig)

	var configMap map[string]interface{}
	if originalConfig != "" {
		err := json.Unmarshal([]byte(originalConfig), &configMap)
		require.NoError(t, err)
	} else {
		configMap = make(map[string]interface{})
	}

	configMap["max_bulk_size_bytes"] = 500

	modifiedConfig, err := json.Marshal(configMap)
	require.NoError(t, err)
	os.Setenv("ELASTIC_CONNECTION_INFO", string(modifiedConfig))

	client := setupTestClient(t)
	require.NotNil(t, client)
	defer client.Close()

	largeDoc := map[string]interface{}{
		"content": strings.Repeat("a", 200),
	}

	client.Index("blob", projectIDString+"_test_1", largeDoc)
	size1 := client.CurrentBatchSize()
	require.NotEmpty(t, size1)
	require.Less(t, size1, 500, "First document should fit within limit")

	client.Index("blob", projectIDString+"_test_2", largeDoc)
	size2 := client.CurrentBatchSize()

	require.NotEmpty(t, size2)
	require.Less(t, size2, 500, "Batch size should be less than max_bulk_size_bytes")

	err = client.Flush()
	require.NoError(t, err)
	require.Equal(t, 0, client.CurrentBatchSize())
}

func TestRemoveBulkSizeTracking(t *testing.T) {
	originalConfig := os.Getenv("ELASTIC_CONNECTION_INFO")
	defer os.Setenv("ELASTIC_CONNECTION_INFO", originalConfig)

	var configMap map[string]interface{}
	if originalConfig != "" {
		err := json.Unmarshal([]byte(originalConfig), &configMap)
		require.NoError(t, err)
	} else {
		configMap = make(map[string]interface{})
	}

	configMap["max_bulk_size_bytes"] = 1000
	modifiedConfig, err := json.Marshal(configMap)
	require.NoError(t, err)

	os.Setenv("ELASTIC_CONNECTION_INFO", string(modifiedConfig))

	client := setupTestClient(t)
	require.NotNil(t, client)
	defer client.Close()

	require.Equal(t, 0, client.CurrentBatchSize())

	client.Remove("blob", "test_id_1")

	require.NotEmpty(t, client.CurrentBatchSize())

	initialSize := client.CurrentBatchSize()

	client.Remove("blob", "test_id_2")

	require.Greater(t, client.CurrentBatchSize(), initialSize)

	err = client.Flush()
	require.NoError(t, err)
	require.Equal(t, 0, client.CurrentBatchSize())
}

func TestDeleteBulkSizeTracking(t *testing.T) {
	originalConfig := os.Getenv("ELASTIC_CONNECTION_INFO")
	defer os.Setenv("ELASTIC_CONNECTION_INFO", originalConfig)

	var configMap map[string]interface{}
	if originalConfig != "" {
		err := json.Unmarshal([]byte(originalConfig), &configMap)
		require.NoError(t, err)
	} else {
		configMap = make(map[string]interface{})
	}

	configMap["max_bulk_size_bytes"] = 1000
	modifiedConfig, err := json.Marshal(configMap)
	require.NoError(t, err)

	os.Setenv("ELASTIC_CONNECTION_INFO", string(modifiedConfig))

	client := setupTestClient(t)
	require.NotNil(t, client)
	defer client.Close()

	largeDoc := map[string]interface{}{
		"content": strings.Repeat("a", 200),
	}
	client.Index("blob", projectIDString+"_test_1", largeDoc)
	client.Index("blob", projectIDString+"_test_2", largeDoc)
	err = client.Flush()
	require.NoError(t, err)

	deleteParams := &elastic.DeleteParams{
		Index:   configMap["index_name"].(string),
		Routing: projectIDString,
		DocId:   projectIDString + "_test_1",
	}

	client.Delete(deleteParams)
	require.NotEmpty(t, client.CurrentBatchSize())

	initialSize := client.CurrentBatchSize()

	deleteParams2 := &elastic.DeleteParams{
		Index:   configMap["index_name"].(string),
		Routing: projectIDString,
		DocId:   projectIDString + "_test_2",
	}
	client.Delete(deleteParams2)
	require.Greater(t, client.CurrentBatchSize(), initialSize)

	err = client.Flush()
	require.NoError(t, err)
	require.Equal(t, 0, client.CurrentBatchSize())
}

func TestMixedOperationsBulkSizeTracking(t *testing.T) {
	originalConfig := os.Getenv("ELASTIC_CONNECTION_INFO")
	defer os.Setenv("ELASTIC_CONNECTION_INFO", originalConfig)

	var configMap map[string]interface{}
	if originalConfig != "" {
		err := json.Unmarshal([]byte(originalConfig), &configMap)
		require.NoError(t, err)
	} else {
		configMap = make(map[string]interface{})
	}

	configMap["max_bulk_size_bytes"] = 1000
	modifiedConfig, err := json.Marshal(configMap)
	require.NoError(t, err)

	os.Setenv("ELASTIC_CONNECTION_INFO", string(modifiedConfig))

	client := setupTestClient(t)
	require.NotNil(t, client)
	defer client.Close()

	doc := map[string]interface{}{"field": "value"}
	client.Index("blob", "test_id_1", doc)
	size1 := client.CurrentBatchSize()

	client.Remove("blob", "test_id_2")
	size2 := client.CurrentBatchSize()
	require.Greater(t, size2, size1)

	deleteParams := &elastic.DeleteParams{
		Index:   configMap["index_name"].(string),
		Routing: projectIDString,
		DocId:   projectIDString + "_test_2",
	}
	client.Delete(deleteParams)
	size3 := client.CurrentBatchSize()
	require.Greater(t, size3, size2)

	err = client.Flush()
	require.NoError(t, err)
	require.Equal(t, 0, client.CurrentBatchSize())
}

func TestConcurrentOperationsThreadSafety(t *testing.T) {
	originalConfig := os.Getenv("ELASTIC_CONNECTION_INFO")
	defer os.Setenv("ELASTIC_CONNECTION_INFO", originalConfig)

	var configMap map[string]interface{}
	if originalConfig != "" {
		err := json.Unmarshal([]byte(originalConfig), &configMap)
		require.NoError(t, err)
	} else {
		configMap = make(map[string]interface{})
	}

	configMap["max_bulk_size_bytes"] = 5000

	modifiedConfig, err := json.Marshal(configMap)
	require.NoError(t, err)
	os.Setenv("ELASTIC_CONNECTION_INFO", string(modifiedConfig))

	client := setupTestClient(t)
	require.NotNil(t, client)
	defer client.Close()

	const numGoroutines = 10
	const operationsPerGoroutine = 20

	var wg sync.WaitGroup
	wg.Add(numGoroutines)

	errorChan := make(chan error, numGoroutines)

	for i := 0; i < numGoroutines; i++ {
		go func(goroutineID int) {
			defer wg.Done()
			defer func() {
				if r := recover(); r != nil {
					errorChan <- fmt.Errorf("goroutine %d panicked: %v", goroutineID, r)
				}
			}()

			for j := 0; j < operationsPerGoroutine; j++ {
				doc := map[string]interface{}{
					"content":      fmt.Sprintf("content from goroutine %d operation %d", goroutineID, j),
					"goroutine_id": goroutineID,
					"operation_id": j,
				}

				switch j % 4 {
				case 0:
					client.Index("blob", fmt.Sprintf("%s_concurrent_%d_%d", projectIDString, goroutineID, j), doc)
				case 1:
					client.Remove("blob", fmt.Sprintf("%s_remove_%d_%d", projectIDString, goroutineID, j))
				case 2:
					client.Delete(&elastic.DeleteParams{
						Index:   client.IndexNameDefault,
						Routing: fmt.Sprintf("project_%v", client.ProjectID),
						DocId:   fmt.Sprintf("delete_%d_%d", goroutineID, j),
					})
				case 3:
					if j%10 == 0 {
						client.Flush()
					} else {
						client.Index("blob", fmt.Sprintf("%s_index_%d_%d", projectIDString, goroutineID, j), doc)
					}
				}

				if j%5 == 0 {
					time.Sleep(1 * time.Millisecond)
				}
			}
		}(i)
	}

	wg.Wait()
	close(errorChan)

	for err := range errorChan {
		t.Errorf("Concurrent operation failed: %v", err)
	}

	require.NoError(t, client.Flush())
	require.Equal(t, 0, client.CurrentBatchSize())
}

func TestConcurrentFlushOperations(t *testing.T) {
	originalConfig := os.Getenv("ELASTIC_CONNECTION_INFO")
	defer os.Setenv("ELASTIC_CONNECTION_INFO", originalConfig)

	var configMap map[string]interface{}
	if originalConfig != "" {
		err := json.Unmarshal([]byte(originalConfig), &configMap)
		require.NoError(t, err)
	} else {
		configMap = make(map[string]interface{})
	}

	configMap["max_bulk_size_bytes"] = 2000

	modifiedConfig, err := json.Marshal(configMap)
	require.NoError(t, err)
	os.Setenv("ELASTIC_CONNECTION_INFO", string(modifiedConfig))

	client := setupTestClient(t)
	require.NotNil(t, client)
	defer client.Close()

	doc := map[string]interface{}{"content": "test content"}
	for i := 0; i < 5; i++ {
		client.Index("blob", fmt.Sprintf("%s_flush_test_%d", projectIDString, i), doc)
	}

	const numFlushGoroutines = 5
	var wg sync.WaitGroup
	wg.Add(numFlushGoroutines)

	errorChan := make(chan error, numFlushGoroutines)

	for i := 0; i < numFlushGoroutines; i++ {
		go func(goroutineID int) {
			defer wg.Done()
			defer func() {
				if r := recover(); r != nil {
					errorChan <- fmt.Errorf("flush goroutine %d panicked: %v", goroutineID, r)
				}
			}()

			for j := 0; j < 3; j++ {
				err := client.Flush()
				if err != nil {
					t.Logf("Flush error in goroutine %d: %v", goroutineID, err)
				}
				time.Sleep(10 * time.Millisecond)
			}
		}(i)
	}

	wg.Wait()
	close(errorChan)

	for err := range errorChan {
		if strings.Contains(err.Error(), "panicked") {
			t.Errorf("Concurrent flush operation failed: %v", err)
		}
	}

	require.Equal(t, 0, client.CurrentBatchSize())
}
