# Go Style Guide

This project follows the [Gitaly Go style guide](https://gitlab.com/gitlab-org/gitaly/-/blob/master/STYLE.md). This document describes project-specific conventions and clarifications.

## General Conventions

For general Go conventions (formatting, naming, imports, testing, etc.), refer to the [Gitaly style guide](https://gitlab.com/gitlab-org/gitaly/-/blob/master/STYLE.md).

### Project-Specific Notes

- **Formatting**: Use `goimports` - run `make format` after changes
- **Testing**: Use `testify/require` for assertions
- **Types**: Use strong types and interfaces; specify int8/16 for size-critical fields when dealing with binary protocols or memory-critical structures

## Error Handling

Follow the [Gitaly error handling guidelines](https://gitlab.com/gitlab-org/gitaly/-/blob/master/STYLE.md#errors), which emphasize:

- Use `%w` when wrapping errors
- Keep errors short
- Use lower case
- Stick to the facts - describe what action you were taking, not your interpretation
- Use `%q` when interpolating strings

### Examples

```go
// Good - describes the action, short and factual
fmt.Errorf("peek diff line: %w", err)
fmt.Errorf("create elasticsearch client: %w", err)
fmt.Errorf("flush bulk operations: %w", err)

// Avoid - breadcrumb style (package.function)
fmt.Errorf("elasticsearch.Connect: %w", err)
fmt.Errorf("codeChunker.ChunkFiles: %w", err)

// Avoid - redundant "failed to" when wrapping
// This creates stutter: "failed to start service: failed to load config: failed to open file"
fmt.Errorf("failed to create client: %w", err)
```

### When Wrapping Errors

When wrapping an error from another function, describe what you were doing, not what failed. The wrapped error already contains the failure information.

```go
// Generating a new error - "failed to" is appropriate
if repository == nil {
    return fmt.Errorf("failed to find repository")
}

// Wrapping an error - just describe the action
if err := loadConfig(); err != nil {
    return fmt.Errorf("load configuration: %w", err)
}
```

### Standard Library IO Errors

Go's standard library IO errors already include the path and operation. Adding these again creates redundancy:

```go
// Avoid - path is already in os.Open error
if err := os.Open(path); err != nil {
    return fmt.Errorf("open file %s: %w", path, err)
}

// Preferred - let stdlib error provide details
if err := os.Open(path); err != nil {
    return fmt.Errorf("open config file: %w", err)
```

## Logging

Refer to the [Gitaly logging guidelines](https://gitlab.com/gitlab-org/gitaly/-/blob/master/STYLE.md#logging) for general patterns.

### Project-Specific Logging

> **Note**: This project is migrating to `slog` (Go's standard structured logging library) across all modes. New code should prefer `slog` where possible.

#### Chunk Mode (uses `slog`)

Use Go's standard `log/slog` package for structured logging:

```go
slog.Debug("indexing chunks", "count", len(chunks), "projectID", projectID)
slog.Info("resolve_reindexing deleted documents", "deleted", result.Deleted)
slog.Error("bulk operation completed with errors", "failed", len(resp.Failed()))
```

Key patterns:
- Use `slog.Debug()` for detailed diagnostic information
- Use `slog.Info()` for important operational events
- Use `slog.Error()` for error conditions
- Add structured fields as alternating key-value pairs after the message

#### Advanced Mode (currently uses `logkit`, migrating to `slog`)

Currently uses `logkit` for structured logging:

```go
logkit.WithError(err).Error("process file")
logkit.WithField("file", filename).Info("processing file")
logkit.WithFields(logkit.Fields{
    "projectID": projectID,
    "groupID":   groupID,
}).Info("indexing project")
```

**Migration note**: New code in advanced mode should use `slog` following the chunk mode patterns above. The existing `logkit` usage will be migrated in a future change.
