package agentcfg

import (
	"testing"

	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v18/internal/tool/testing/testhelpers"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v18/internal/tool/testing/testproto"
	"google.golang.org/protobuf/types/known/structpb"
)

func TestValidation_Valid(t *testing.T) {
	tests := []testhelpers.ValidTestcase{
		{
			Name:  "empty",
			Valid: &AgentConfiguration{},
		},
		{
			Name: "empty CiAccessGroupCF.DefaultNamespace",
			Valid: &CiAccessGroupCF{
				Id:               "abc",
				DefaultNamespace: "", // empty is ok
			},
		},
		{
			Name: "empty environments CiAccessGroupCF.Environments",
			Valid: &CiAccessGroupCF{
				Id:           "abc",
				Environments: []string{},
			},
		},
		{
			Name: "multiple environments CiAccessGroupCF.Environments",
			Valid: &CiAccessGroupCF{
				Id:           "abc",
				Environments: []string{"a", "b"},
			},
		},
		{
			Name: "empty CiAccessGroupCF.Annotations",
			Valid: &CiAccessGroupCF{
				Id:          "abc",
				Annotations: &structpb.Struct{},
			},
		},
		{
			Name: "CiAccessGroupCF.Annotations exists",
			Valid: &CiAccessGroupCF{
				Id: "abc",
				Annotations: &structpb.Struct{
					Fields: map[string]*structpb.Value{
						"key1": {
							Kind: &structpb.Value_StringValue{
								StringValue: "value1",
							},
						},
						"key2": {
							Kind: &structpb.Value_StringValue{
								StringValue: "value2",
							},
						},
					},
				},
			},
		},
		{
			Name: "empty environments CiAccessProjectCF.Environments",
			Valid: &CiAccessProjectCF{
				Id:           "abc",
				Environments: []string{},
			},
		},
		{
			Name: "multiple environments CiAccessProjectCF.Environments",
			Valid: &CiAccessProjectCF{
				Id:           "abc",
				Environments: []string{"a", "b"},
			},
		},
		{
			Name: "empty CiAccessProjectCF.Annotations",
			Valid: &CiAccessProjectCF{
				Id:          "abc",
				Annotations: &structpb.Struct{},
			},
		},
		{
			Name: "CiAccessProjectCF.Annotations exists",
			Valid: &CiAccessProjectCF{
				Id: "abc",
				Annotations: &structpb.Struct{
					Fields: map[string]*structpb.Value{
						"key1": {
							Kind: &structpb.Value_StringValue{
								StringValue: "value1",
							},
						},
						"key2": {
							Kind: &structpb.Value_StringValue{
								StringValue: "value2",
							},
						},
					},
				},
			},
		},
		{
			Name:  "empty CiAccessAsAgentCF",
			Valid: &CiAccessAsAgentCF{},
		},
		{
			Name:  "empty CiAccessAsCiJobCF",
			Valid: &CiAccessAsCiJobCF{},
		},
		{
			Name:  "empty CiAccessAsCiJobCF",
			Valid: &CiAccessAsCiJobCF{},
		},
		{
			Name:  "empty CiAccessAsAgentCF",
			Valid: &CiAccessAsAgentCF{},
		},
		{
			Name: "minimal CiAccessAsImpersonateCF",
			Valid: &CiAccessAsImpersonateCF{
				Username: "abc",
			},
		},
		{
			Name: "one group CiAccessAsImpersonateCF",
			Valid: &CiAccessAsImpersonateCF{
				Username: "abc",
				Groups:   []string{"g"},
			},
		},
		{
			Name:  "minimal RemoteDevelopmentCF",
			Valid: &RemoteDevelopmentCF{},
		},
	}
	testhelpers.AssertValid(t, tests)
}

func TestValidation_Invalid(t *testing.T) {
	emptyID := ""
	tests := []testhelpers.InvalidTestcase{
		{
			ErrString: "validation error: id: value length must be at least 1 bytes",
			Invalid: &CiAccessGroupCF{
				Id: emptyID, // empty id is not ok
			},
		},
		{
			ErrString: "validation error: environments[0]: value length must be at least 1 bytes",
			Invalid: &CiAccessGroupCF{
				Id:           "abc",
				Environments: []string{""},
			},
		},
		{
			ErrString: "validation error: environments[0]: value length must be at least 1 bytes",
			Invalid: &CiAccessProjectCF{
				Id:           "abc",
				Environments: []string{""},
			},
		},
		{
			ErrString: "validation error: as: exactly one field is required in oneof",
			Invalid:   &CiAccessAsCF{},
		},
		{
			ErrString: "validation error: impersonate.username: value length must be at least 1 bytes",
			Invalid: &CiAccessAsCF{
				As: &CiAccessAsCF_Impersonate{},
			},
		},
		{
			ErrString: "validation error: username: value length must be at least 1 bytes",
			Invalid:   &CiAccessAsImpersonateCF{},
		},
		{
			ErrString: "validation error: groups[0]: value length must be at least 1 bytes",
			Invalid: &CiAccessAsImpersonateCF{
				Username: "a",
				Groups:   []string{""},
			},
		},
		{
			ErrString: "validation errors:\n - key: value length must be at least 1 bytes\n - val: value must contain at least 1 item(s)",
			Invalid:   &ExtraKeyValCF{},
		},
	}
	testhelpers.AssertInvalid(t, tests)
}

func TestJSONNameSet(t *testing.T) {
	cfg := ConfigurationFile{}
	testproto.AssertHasJSONNameSetRecursive(t, cfg.ProtoReflect().Descriptor())
}
