package server_api //nolint:staticcheck

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v18/internal/api"
)

var (
	_ AgentInfo = (*AgentkInfo)(nil)
	_ AgentInfo = (*AgentwInfo)(nil)
	_ AgentInfo = (*RunnerControllerInfo)(nil)
)

func TestAPI_AgentToken2Key(t *testing.T) {
	// GIVEN
	tests := []struct {
		name        string
		token       api.AgentToken
		expectedKey string
	}{
		{
			name:        "empty token",
			token:       "",
			expectedKey: "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855", // SHA256 of empty string
		},
		{
			name:        "very short token with 5 chars",
			token:       "abcde",
			expectedKey: "36bbe50ed96841d10443bcb670d6554f0a34b761be67ec9c4a8ad2c0c44ca42c", // SHA256 of "abcde"
		},
		{
			name:        "token exactly 20 chars",
			token:       "abcdefghij1234567890",
			expectedKey: "401cc435d05bda0fb77f7abff777681398a0e058b17d439f2cb3847ad9bf9f6d", // SHA256 of first 20 chars
		},
		{
			name:        "token 30 chars",
			token:       "abcdefghij1234567890abcdefghij",
			expectedKey: "401cc435d05bda0fb77f7abff777681398a0e058b17d439f2cb3847ad9bf9f6d", // SHA256 of first 20 chars
		},
		{
			// see https://handbook.gitlab.com/handbook/engineering/architecture/design-documents/cells/routable_tokens/#minimum-token-length
			name:        "token exactly 40 chars (minimum routable token length)",
			token:       "abcdefghij1234567890abcdefghij1234567890",
			expectedKey: "401cc435d05bda0fb77f7abff777681398a0e058b17d439f2cb3847ad9bf9f6d", // SHA256 of first 20 chars
		},
		{
			name:        "token 50 chars",
			token:       "abcdefghij1234567890abcdefghij1234567890abcdefghij",
			expectedKey: "5061859e4a9d0217ee86442eedc51d9ede38bedf2ff79c5b7d3032f37ba7614a", // SHA256 of first 25 chars
		},
		{
			// see https://handbook.gitlab.com/handbook/engineering/architecture/design-documents/cells/routable_tokens/#maximum-token-length
			name:        "token 333 chars (maximum routable token length)",
			token:       "abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij1234567890abcdefghij123",
			expectedKey: "1f04e3f233eee241981edda9cb24bdea2044055ba9fcfcf278f938e15a043496", // SHA256 of first 166 chars
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			// WHEN
			actualKey := AgentToken2key(tt.token)

			// THEN
			assert.Equal(t, tt.expectedKey, fmt.Sprintf("%x", actualKey))
		})
	}
}
