package agentk

import (
	"context"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v18/internal/module/modagentk"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v18/internal/tool/testing/testlogger"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v18/pkg/agentcfg"
	"gitlab.com/gitlab-org/cluster-integration/tunnel/tool/retry"
	"k8s.io/apimachinery/pkg/util/wait"
)

var (
	_ modagentk.Module  = &module{}
	_ modagentk.Factory = &Factory{}
)

type mockReconciler struct {
	timesCalled     uint32
	fullInterval    time.Duration
	partialInterval time.Duration
	runError        error
}

func (r *mockReconciler) Stop() {
	// do nothing
}

func (r *mockReconciler) Run(_ context.Context) (WorkerSettings, error) {
	r.timesCalled += 1

	return WorkerSettings{r.fullInterval, r.partialInterval}, r.runError
}

func TestConfigChange(t *testing.T) {
	tests := []struct {
		description string
		timesCalled uint32
		configs     []*agentcfg.AgentConfiguration
	}{
		{
			description: "When remote dev is enabled in the config, does start reconciler",
			timesCalled: 1,
			configs: []*agentcfg.AgentConfiguration{
				{
					RemoteDevelopment: &agentcfg.RemoteDevelopmentCF{
						Enabled: true,
					},
				},
			},
		},
		{
			description: "When the config is updated, restarts the reconciler",
			timesCalled: 2,
			configs: []*agentcfg.AgentConfiguration{
				{
					RemoteDevelopment: &agentcfg.RemoteDevelopmentCF{
						Enabled: true,
					},
				},
				{
					RemoteDevelopment: &agentcfg.RemoteDevelopmentCF{
						Enabled: false,
					},
				},
				{
					RemoteDevelopment: &agentcfg.RemoteDevelopmentCF{
						Enabled: true,
					},
				},
			},
		},
		{
			description: "When the config is published multiple times without any changes",
			timesCalled: 1,
			configs: []*agentcfg.AgentConfiguration{
				{
					RemoteDevelopment: &agentcfg.RemoteDevelopmentCF{
						Enabled: true,
					},
				},
				{
					RemoteDevelopment: &agentcfg.RemoteDevelopmentCF{
						Enabled: true,
					},
				},
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.description, func(t *testing.T) {
			configChannel := make(chan *agentcfg.AgentConfiguration)
			mock := &mockReconciler{
				fullInterval:    1 * time.Hour,
				partialInterval: 10 * time.Second,
			}

			mod := module{
				log: testlogger.New(t),
				api: newMockAPI(t),
				prerequisitesPollCfg: retry.PollConfig{
					Interval: defaultFullReconciliationInterval,
					Backoff: retry.NewExponentialBackoff(
						initBackoff,
						defaultFullReconciliationInterval*2,
						resetDuration,
						backoffFactor,
						jitter,
					),
				},
				reconcilerFactory: func(ctx context.Context, namespace string) (remoteDevReconciler, error) {
					return mock, nil
				},
			}

			ctx := context.Background()

			wg := wait.Group{}

			// publish configs asynchronously
			wg.StartWithContext(ctx, func(ctx context.Context) {
				publishInterval := 50 * time.Millisecond

				if tt.configs != nil {
					for _, cfg := range tt.configs {
						// populate the test config with defaults if missing
						// this must be explicitly done in tests where module's Run() is invoked directly
						err := mod.DefaultAndValidateConfiguration(cfg)
						require.NoError(t, err)

						configChannel <- cfg
						time.Sleep(publishInterval)
					}
				}
				close(configChannel)
			})

			err := mod.Run(ctx, configChannel)
			wg.Wait()
			require.NoError(t, err)
			require.Equal(t, tt.timesCalled, mock.timesCalled)
		})
	}
}
