package runner_controller //nolint:staticcheck

import (
	"context"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v18/internal/api"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v18/internal/module/runner_controller/rpc"
	"gitlab.com/gitlab-org/cluster-integration/tunnel/tool/grpcz"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
	"k8s.io/apimachinery/pkg/util/wait"
)

// testRunnerControllerServer is a test implementation of RunnerControllerServiceServer
type testRunnerControllerServer struct {
	rpc.UnimplementedRunnerControllerServiceServer
	admittedValue bool
}

func (s *testRunnerControllerServer) Admit(ctx context.Context, req *rpc.AdmitRequest) (*rpc.AdmitResponse, error) {
	return &rpc.AdmitResponse{
		Admitted: s.admittedValue,
	}, nil
}

func TestRunnerControllerClient_Admit(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name          string
		admittedValue bool
	}{
		{
			name:          "admits successfully",
			admittedValue: true,
		},
		{
			name:          "rejects successfully",
			admittedValue: false,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			// Create the test server and get the gRPC client connection
			conn := setupRunnerControllerServer(t, tt.admittedValue)

			// Create the RunnerControllerClient with a connection pool that returns our test connection
			agentKey := api.AgentKey{
				ID:   12345,
				Type: api.AgentTypeRunnerController,
			}
			client := NewRunnerControllerClient(func(key api.AgentKey) grpc.ClientConnInterface {
				assert.Equal(t, agentKey, key)
				return conn
			})

			resp, err := client.Admit(t.Context(), agentKey, &rpc.AdmitRequest{})
			require.NoError(t, err)

			assert.Equal(t, tt.admittedValue, resp.Admitted)
		})
	}
}

func setupRunnerControllerServer(t *testing.T, admittedValue bool) *grpc.ClientConn {
	t.Helper()

	// Create pipe listener for in-memory connection
	l := grpcz.NewPipeDialListener()
	t.Cleanup(func() {
		assert.NoError(t, l.Close())
	})

	// Create the test server implementation
	testServer := &testRunnerControllerServer{
		admittedValue: admittedValue,
	}

	// Create gRPC server and register service
	s := grpc.NewServer()
	rpc.RegisterRunnerControllerServiceServer(s, testServer)

	// Start server in background
	var wg wait.Group
	t.Cleanup(func() {
		s.GracefulStop()
		wg.Wait()
	})
	wg.Start(func() {
		assert.NoError(t, s.Serve(l))
	})

	// Create client connection using the pipe dialer
	conn, err := grpc.NewClient("passthrough:pipe",
		grpc.WithContextDialer(l.DialContext),
		grpc.WithTransportCredentials(insecure.NewCredentials()),
	)
	require.NoError(t, err)
	t.Cleanup(func() {
		assert.NoError(t, conn.Close())
	})

	return conn
}
