package modshared

import (
	"context"
	"log/slog"

	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v18/internal/api"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v18/internal/tool/errz"
	"gitlab.com/gitlab-org/cluster-integration/tunnel/tool/retry"
)

var (
	NoAgentKey = api.AgentKey{ID: 0, Type: api.AgentTypeUnknown}
)

type Factory interface {
	// Name returns module's name.
	Name() string
}

// API provides the API for the module to use.
type API interface {
	// HandleProcessingError can be used to handle errors occurring while processing a request.
	// If err is a (or wraps a) errz.UserError, it might be handled specially.
	HandleProcessingError(ctx context.Context, log *slog.Logger, msg string, err error, fields ...slog.Attr)
}

// RPCAPI provides the API for the module's gRPC handlers to use.
type RPCAPI interface {
	// Log returns a logger to use in the context of the request being processed.
	Log() *slog.Logger
	// HandleProcessingError can be used to handle errors occurring while processing a request.
	// If err is a (or wraps a) errz.UserError, it might be handled specially.
	HandleProcessingError(log *slog.Logger, msg string, err error, fields ...slog.Attr)
	// HandleIOError can be used to handle I/O error produced by gRPC Send(), Recv() methods or any other I/O error.
	// It returns an error, compatible with gRPC status package.
	HandleIOError(log *slog.Logger, msg string, err error) error
	// PollWithBackoff runs f every duration given by BackoffManager.
	//
	// PollWithBackoff should be used by the top-level polling, so that it can be gracefully interrupted
	// by the server when necessary. E.g. when stream is nearing it's max connection age or program needs to
	// be shut down.
	// If sliding is true, the period is computed after f runs. If it is false then
	// period includes the runtime for f.
	// It returns when:
	// - stream's context is canceled or max connection age has been reached. nil is returned in this case.
	// - f returns Done. error from f is returned in this case.
	PollWithBackoff(cfg retry.PollConfig, f retry.PollWithBackoffFunc, opts ...retry.PollOption) error
}

func APIToErrReporter(api API) errz.ErrReporter {
	return &errReporter{
		API: api,
	}
}

type errReporter struct {
	API API
}

func (r *errReporter) HandleProcessingError(ctx context.Context, log *slog.Logger, msg string, err error, fields ...slog.Attr) {
	r.API.HandleProcessingError(ctx, log, msg, err, fields...)
}
