package watch_graph //nolint:staticcheck
import (
	"fmt"

	"k8s.io/apimachinery/pkg/runtime/schema"
)

const (
	InfSyncFailedWarnType          = "INFORMER_SYNC_FAILED"
	DiscoFailedWarnType            = "DISCOVERY_FAILED"
	NamespaceListFailedWarnType    = "NAMESPACE_LIST_FAILED"
	ObjectProcessingFailedWarnType = "OBJECT_PROCESSING_FAILED"
	InternalErrorWarnType          = "INTERNAL_ERROR"
)

type Warning struct {
	Type       string
	Message    string
	Attributes WarningAttrs
}

// WarningAttrs holds attributes for all warning types.
// All fields must be marked omitempty because different warnings have different fields set.
type WarningAttrs struct {
	Group     string `json:"g,omitempty"`
	Version   string `json:"v,omitempty"`
	Resource  string `json:"r,omitempty"`
	Namespace string `json:"ns,omitempty"`
	Name      string `json:"n,omitempty"`
}

func (a *WarningAttrs) IsZero() bool {
	var zero WarningAttrs
	return *a == zero
}

func NewObjectProcessingWarning(gvr schema.GroupVersionResource, namespace, name, message string) Warning {
	return Warning{
		Type:    ObjectProcessingFailedWarnType,
		Message: message,
		Attributes: WarningAttrs{
			Group:     gvr.Group,
			Version:   gvr.Version,
			Resource:  gvr.Resource,
			Namespace: namespace,
			Name:      name,
		},
	}
}

func NewNamespaceListFailedWarning(namespace string, err error) Warning {
	return Warning{
		Type:    NamespaceListFailedWarnType,
		Message: fmt.Sprintf("failed to get namespace %q: %v", namespace, err),
		Attributes: WarningAttrs{
			Namespace: namespace,
		},
	}
}
