package rpc

import (
	"testing"

	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v18/internal/tool/testing/testhelpers"
)

func TestValidation_Valid(t *testing.T) {
	arcTypes := []string{"or", "r", "t"}
	tests := []testhelpers.ValidTestcase{
		{
			Name:  "Namespaces minimal",
			Valid: &Namespaces{},
		},
		{
			Name: "Namespaces explicit list",
			Valid: &Namespaces{
				Names: []string{"n1"},
			},
		},
		{
			Name: "Namespaces explicit list and CEL",
			Valid: &Namespaces{
				Names:                    []string{"n1"},
				ObjectSelectorExpression: "s1",
			},
		},
		{
			Name: "Namespaces all with label filtering",
			Valid: &Namespaces{
				LabelSelector: "l=v",
			},
		},
		{
			Name: "Namespaces all with field filtering",
			Valid: &Namespaces{
				FieldSelector: "f=v",
			},
		},
		{
			Name: "Namespaces all with label and field filtering and CEL",
			Valid: &Namespaces{
				LabelSelector:            "l=v",
				FieldSelector:            "f=v",
				ObjectSelectorExpression: "s1",
			},
		},
	}
	for _, arcType := range arcTypes {
		tests = append(tests, testhelpers.ValidTestcase{
			Name: "Action_SetArc minimal " + arcType,
			Valid: &Action_SetArc{
				Source: &Action_Vertex{
					Version:  "v1",
					Resource: "clusterroles",
					Name:     "admin",
				},
				Destination: &Action_Vertex{
					Version:  "v1",
					Resource: "clusterrolebindings",
					Name:     "admin",
				},
				Type: arcType,
			},
		})
	}
	testhelpers.AssertValid(t, tests)
}

func TestValidation_Invalid(t *testing.T) {
	tests := []testhelpers.InvalidTestcase{
		{
			ErrString: "validation error:\n - source: value is required [required]\n - destination: value is required [required]\n - type: invalid arc type [string.is_arc_type]",
			Invalid:   &Action_SetArc{},
		},
		{
			ErrString: "validation error:\n - source: value is required [required]\n - destination: value is required [required]\n - type: invalid arc type [string.is_arc_type]",
			Invalid: &Action_SetArc{
				Type: "some rubbish",
			},
		},
		{
			ErrString: "validation error:\n - source: value is required [required]\n - destination: value is required [required]\n - type: invalid arc type [string.is_arc_type]",
			Invalid:   &Action_DeleteArc{},
		},
		{
			ErrString: "validation error:\n - source: value is required [required]\n - destination: value is required [required]\n - type: invalid arc type [string.is_arc_type]",
			Invalid: &Action_DeleteArc{
				Type: "some rubbish",
			},
		},
		{
			ErrString: "validation error:\n - either names or selectors can be use, not both [names_xor_labels_fields]",
			Invalid: &Namespaces{
				Names:         []string{"n1"},
				LabelSelector: "l=v",
			},
		},
		{
			ErrString: "validation error:\n - either names or selectors can be use, not both [names_xor_labels_fields]",
			Invalid: &Namespaces{
				Names:         []string{"n1"},
				FieldSelector: "f=v",
			},
		},
	}
	testhelpers.AssertInvalid(t, tests)
}
