package tunclient

import (
	"context"
	"sync"
	"sync/atomic"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

var (
	_ ConnectionInterface = (*mockConnection)(nil)
)

func TestConnManager_StartsMinIdleConnectionsOnRun(t *testing.T) {
	cm, conns, mu := setupConnManager(t)
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	go cm.Run(ctx)
	require.Eventually(t, func() bool {
		cm.mu.Lock()
		defer cm.mu.Unlock()
		mu.Lock()
		defer mu.Unlock()
		return len(*conns) == int(cm.minIdleConnections)
	}, time.Minute, 10*time.Millisecond)
	cancel()
	cm.wg.Wait()
	require.Len(t, *conns, int(cm.minIdleConnections))
}

func TestConnManager_ScalesIdleConnectionsToMaxAndThenToMin(t *testing.T) {
	cm, conns, mu := setupConnManager(t)
	cm.maxIdleTime = 50 * time.Millisecond
	var activated int
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	go cm.Run(ctx)
	// Scale to max
	require.Eventually(t, func() bool {
		mu.Lock()
		lenConns := len(*conns)
		assert.LessOrEqual(t, lenConns, int(cm.maxConnections))
		toActivate := make([]*mockConnection, lenConns-activated)
		copy(toActivate, (*conns)[activated:])
		mu.Unlock()
		for _, c := range toActivate { // activate any new connections, must not hold the mutex
			activated++
			c.onActive(c)
		}
		return lenConns == int(cm.maxConnections)
	}, time.Minute, 10*time.Millisecond)
	// Scale to min
	cm.mu.Lock()
	cns := make([]ConnectionInterface, 0, len(cm.connections))
	for c, i := range cm.connections {
		if i.state == active {
			cns = append(cns, c)
		}
	}
	cm.mu.Unlock()
	for _, c := range cns {
		c.(*mockConnection).onIdle(c)
	}
	time.Sleep(cm.maxIdleTime + 10*time.Millisecond)
	for _, c := range cns {
		c.(*mockConnection).onIdle(c)
	}
	require.Eventually(t, func() bool { // poll while goroutines are shutting down
		cm.mu.Lock()
		defer cm.mu.Unlock()
		return cm.idleConnections == cm.minIdleConnections && cm.activeConnections == 0 && len(cm.connections) == int(cm.minIdleConnections)
	}, time.Minute, 10*time.Millisecond)
	cancel()
	cm.wg.Wait()
	require.Len(t, *conns, int(cm.maxConnections))
}

func setupConnManager(t *testing.T) (*ConnectionManager, *[]*mockConnection, *sync.Mutex) {
	t.Parallel()
	var conns []*mockConnection
	var mu sync.Mutex
	cm := &ConnectionManager{
		connections:        make(map[ConnectionInterface]connectionInfo),
		minIdleConnections: 1,
		maxConnections:     100,
		scaleUpStep:        2,
		maxIdleTime:        time.Minute,
		connectionFactory: func(onActive, onIdle func(ConnectionInterface)) ConnectionInterface {
			c := &mockConnection{
				onActive: onActive,
				onIdle:   onIdle,
			}
			mu.Lock()
			defer mu.Unlock()
			conns = append(conns, c)
			return c
		},
	}
	t.Cleanup(func() {
		cm.wg.Wait()
		assert.Zero(t, cm.idleConnections)
		assert.Zero(t, cm.activeConnections)
		assert.Empty(t, cm.connections)
		for _, c := range conns {
			assert.EqualValues(t, 1, c.runCalled.Load())
			assert.EqualValues(t, 1, c.stopped.Load())
		}
	})
	return cm, &conns, &mu
}

type mockConnection struct {
	runCalled, stopped atomic.Int32
	onActive, onIdle   func(ConnectionInterface)
}

func (m *mockConnection) Run(attemptCtx, pollCtx context.Context) {
	defer m.stopped.Add(1)
	m.runCalled.Add(1)
	<-pollCtx.Done()
}
