package api

import (
	"net/http"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v17/internal/tool/testing/mock_gitlab"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v17/internal/tool/testing/testhelpers"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v17/pkg/entity"
)

func TestGetProjectInfo(t *testing.T) {
	const (
		projectID = "bla/bla"
	)
	ctx, traceID := testhelpers.CtxWithSpanContext(t)
	response := &GetProjectInfoResponse{
		ProjectId: 234,
		GitalyInfo: &entity.GitalyInfo{
			Address: "example.com",
			Token:   "123123",
			Features: map[string]string{
				"a": "b",
			},
		},
		GitalyRepository: &entity.GitalyRepository{
			StorageName:                   "234",
			RelativePath:                  "123",
			GitObjectDirectory:            "sfasdf",
			GitAlternateObjectDirectories: []string{"a", "b"},
			GlRepository:                  "254634",
			GlProjectPath:                 "64662",
		},
		DefaultBranch: "main",
	}
	gitLabClient := mock_gitlab.SetupClient(t, ProjectInfoAPIPath, func(w http.ResponseWriter, r *http.Request) {
		testhelpers.AssertRequestMethod(t, r, http.MethodGet)
		testhelpers.AssertGetJSONRequestIsCorrect(t, r, traceID)
		assert.Equal(t, projectID, r.URL.Query().Get(ProjectIDQueryParam))

		testhelpers.RespondWithJSON(t, w, response)
	})

	projInfo, err := GetProjectInfo(ctx, gitLabClient, testhelpers.AgentkToken, projectID)
	require.NoError(t, err)

	assert.Equal(t, response.ProjectId, projInfo.ProjectID)
	AssertGitalyInfo(t, response.GitalyInfo, projInfo.GitalyInfo)
	AssertGitalyRepository(t, response.GitalyRepository, projInfo.Repository)
	assert.Equal(t, response.DefaultBranch, projInfo.DefaultBranch)
}
