package agentkapp

import (
	"errors"
	"net"

	"github.com/ash2k/stager"
	"github.com/bufbuild/protovalidate-go"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v17/internal/api"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v17/internal/module/modshared"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v17/internal/tool/grpctool"
	"go.opentelemetry.io/contrib/instrumentation/google.golang.org/grpc/otelgrpc"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
)

// inMemAPIServer represents agentk API that kas can talk to when running in agentk->kas tunnel mode.
type inMemAPIServer struct {
	server        *grpc.Server
	inMemConn     *grpc.ClientConn
	inMemListener net.Listener
}

func newInMemAPIServer(ot *obsTools, factory modshared.RPCAPIFactory, v protovalidate.Validator) (*inMemAPIServer, error) {
	// In-mem gRPC client->listener pipe
	listener := grpctool.NewDialListener()

	// Construct connection to the API gRPC server
	conn, err := grpc.NewClient("passthrough:api-server",
		grpc.WithSharedWriteBuffer(true),
		grpc.WithContextDialer(listener.DialContext),
		grpc.WithTransportCredentials(insecure.NewCredentials()),
		grpc.WithDefaultCallOptions(grpc.MaxCallRecvMsgSize(api.GRPCMaxMessageSize)),
	)
	if err != nil {
		return nil, err
	}
	return &inMemAPIServer{
		server: grpc.NewServer(
			grpc.StatsHandler(otelgrpc.NewServerHandler(
				otelgrpc.WithTracerProvider(ot.tp),
				otelgrpc.WithMeterProvider(ot.mp),
				otelgrpc.WithPropagators(ot.p),
				otelgrpc.WithMessageEvents(otelgrpc.ReceivedEvents, otelgrpc.SentEvents),
			)),
			grpc.StatsHandler(grpctool.ServerNoopMaxConnAgeStatsHandler{}),
			grpc.SharedWriteBuffer(true),
			grpc.ChainStreamInterceptor(
				ot.streamProm, // 1. measure all invocations
				modshared.StreamRPCAPIInterceptor(factory),    // 2. inject RPC API
				grpctool.StreamServerValidatingInterceptor(v), // x. wrap with validator
			),
			grpc.ChainUnaryInterceptor(
				ot.unaryProm, // 1. measure all invocations
				modshared.UnaryRPCAPIInterceptor(factory),    // 2. inject RPC API
				grpctool.UnaryServerValidatingInterceptor(v), // x. wrap with validator
			),
			grpc.MaxRecvMsgSize(api.GRPCMaxMessageSize),
		),
		inMemConn:     conn,
		inMemListener: listener,
	}, nil
}

func (s *inMemAPIServer) Start(stage stager.Stage) {
	grpctool.StartServer(stage, s.server,
		func() (net.Listener, error) {
			return s.inMemListener, nil
		},
		func() {},
		func() {},
	)
}

func (s *inMemAPIServer) Close() error {
	return errors.Join(
		s.inMemConn.Close(),     // first close the client
		s.inMemListener.Close(), // then close the listener (if not closed already)
	)
}
