import fs from 'fs/promises';
import path from 'path';

// Expected artifact names from .gitlab-ci.yml
const EXPECTED_ARTIFACTS = [
  'linux-x64-gnu',
  'linux-arm64-gnu',
  'linux-x64-musl',
  'linux-arm64-musl',
  'linux-arm-gnueabihf',
  'darwin-x64',
  'darwin-arm64',
  'win32-x64-msvc',
  'win32-ia32-msvc',
  'win32-arm64-msvc'
];

// Function to read index.js and extract artifact names
async function extractArtifactNamesFromIndex(): Promise<string[]> {
  const indexPath = path.join(__dirname, '..', 'packages', 'gitalisk-node', 'index.js');
  const content = await fs.readFile(indexPath, 'utf-8');
  
  // Extract artifact names from require statements
  const artifactNames: string[] = [];
  const requireRegex = /require\('\.\/gitalisk-node\.([^.]+)\.node'\)/g;
  let match;
  
  while ((match = requireRegex.exec(content)) !== null) {
    artifactNames.push(match[1]);
  }
  
  return artifactNames;
}

// Function to verify artifacts exist in packages directory
async function verifyArtifactsExist(): Promise<void> {
  const packagesDir = path.join(__dirname, '..', 'packages', 'gitalisk-node');
  
  // Check each expected artifact
  const missingArtifacts = await Promise.all(
    EXPECTED_ARTIFACTS.map(async (artifact) => {
      const artifactPath = path.join(packagesDir, `gitalisk-node.${artifact}.node`);
      try {
        await fs.access(artifactPath);
        return null;
      } catch {
        return artifact;
      }
    })
  ).then(results => results.filter(Boolean) as string[]);
  
  if (missingArtifacts.length > 0) {
    console.error('❌ Missing artifacts:');
    missingArtifacts.forEach(artifact => {
      console.error(`  - ${artifact}`);
    });
    process.exit(1);
  }
  
  console.log('✅ All required artifacts exist');
}

// Main execution
async function main() {
  try {
    const foundArtifacts = await extractArtifactNamesFromIndex();
    console.log('Found artifacts in index.js:', foundArtifacts);
    
    await verifyArtifactsExist();
  } catch (error) {
    console.error('Error:', error);
    process.exit(1);
  }
}

main();
