#!/bin/bash
set -e

echo "🚀 Running semantic-release prepare script..."

# Get the new version from command line argument
NEW_VERSION="$1"

if [ -z "$NEW_VERSION" ]; then
    echo "❌ Error: NEW_VERSION is not provided as first argument. This script should be run by semantic-release."
    exit 1
fi

echo "📝 Updating version to: $NEW_VERSION"

# Update the VERSION file
echo "$NEW_VERSION" > VERSION
echo "✅ Updated VERSION file"

# Update workspace packages using npm version with workspace flags
echo "📦 Updating npm workspace packages..."

echo "📦 Updating @gitlab-org/gitalisk-node..."
npm version "$NEW_VERSION" --workspace=@gitlab-org/gitalisk-node --git-tag-version=false
echo "✅ Updated @gitlab-org/gitalisk-node to version $NEW_VERSION"

echo "📦 Updating @gitlab-org/gitalisk-node-tests..."
npm version "$NEW_VERSION" --workspace=@gitlab-org/gitalisk-node-tests --git-tag-version=false
echo "✅ Updated @gitlab-org/gitalisk-node-tests to version $NEW_VERSION"

# Note: Root package.json is already updated by @semantic-release/npm plugin

# Update Cargo workspace packages
echo "🦀 Updating Cargo workspace packages..."

# Update root Cargo.toml workspace package version
if [ -f "Cargo.toml" ]; then
    echo "📦 Updating root Cargo.toml workspace package version..."
    sed -i.bak "s/^version = \".*\"/version = \"$NEW_VERSION\"/" Cargo.toml
    rm -f Cargo.toml.bak
    echo "✅ Updated root Cargo.toml workspace package to version $NEW_VERSION"
fi

# Update workspace members manually
if [ -f "Cargo.toml" ]; then
    echo "📦 Updating Cargo workspace members..."
    
    # Update gitalisk-core
    if [ -f "crates/gitalisk-core/Cargo.toml" ]; then
        echo "📦 Updating gitalisk-core version..."
        sed -i.bak "s/^version = \".*\"/version = \"$NEW_VERSION\"/" "crates/gitalisk-core/Cargo.toml"
        rm -f "crates/gitalisk-core/Cargo.toml.bak"
        echo "✅ Updated gitalisk-core to version $NEW_VERSION"
    fi
    
    # Update gitalisk-node
    if [ -f "crates/gitalisk-node/Cargo.toml" ]; then
        echo "📦 Updating gitalisk-node version..."
        sed -i.bak "s/^version = \".*\"/version = \"$NEW_VERSION\"/" "crates/gitalisk-node/Cargo.toml"
        rm -f "crates/gitalisk-node/Cargo.toml.bak"
        echo "✅ Updated gitalisk-node to version $NEW_VERSION"
    fi
    
    # Update Cargo.lock with new workspace versions
    echo "🔄 Updating Cargo.lock..."
    cargo update --workspace
    echo "✅ Updated Cargo.lock"
else
    echo "⚠️  Cargo.toml not found, skipping Cargo updates"
fi

echo "🎉 All versions updated to $NEW_VERSION!"
echo "🔍 Verification:"
echo "   - VERSION file: $(cat VERSION)"
echo "   - Root package.json: $(npm pkg get version | tr -d '\"')"
if [ -f "packages/gitalisk-node/package.json" ]; then
    echo "   - gitalisk-node package.json: $(cd packages/gitalisk-node && npm pkg get version | tr -d '\"')"
fi

echo ""
echo "✅ All npm workspace packages updated with proper workspace commands"
echo "✅ package-lock.json automatically updated by npm version commands"
echo "✅ All Cargo workspace members updated and Cargo.lock synced" 
