#!/bin/sh
set -eu

# Script to check if versions across VERSION file, Cargo.toml and package.json are in sync

VERSION_FILE="VERSION"
WORKSPACE_CARGO="Cargo.toml"
PACKAGE_JSON="package.json"

if [ ! -f "$VERSION_FILE" ]; then
  echo "Error: $VERSION_FILE not found"
  exit 1
fi

if [ ! -f "$WORKSPACE_CARGO" ]; then
  echo "Error: $WORKSPACE_CARGO not found"
  exit 1
fi

if [ ! -f "$PACKAGE_JSON" ]; then
  echo "Error: $PACKAGE_JSON not found"
  exit 1
fi

# Read versions from files
VERSION_FILE_VERSION=$(cat "$VERSION_FILE" | tr -d '[:space:]')
CARGO_WORKSPACE_VERSION=$(grep 'version =' "$WORKSPACE_CARGO" | head -n 1 | cut -d'"' -f2)
PACKAGE_JSON_VERSION=$(grep '"version":' "$PACKAGE_JSON" | head -n 1 | cut -d'"' -f4)

# Display versions
echo "Checking versions across project files:"
echo "  VERSION file:   $VERSION_FILE_VERSION"
echo "  Cargo.toml:     $CARGO_WORKSPACE_VERSION"
echo "  package.json:   $PACKAGE_JSON_VERSION"

EXIT_CODE=0

# Check if all versions match
if [ "$VERSION_FILE_VERSION" != "$CARGO_WORKSPACE_VERSION" ]; then
  echo "❌ Error: VERSION file ($VERSION_FILE_VERSION) does not match Cargo.toml workspace version ($CARGO_WORKSPACE_VERSION)"
  EXIT_CODE=1
fi

if [ "$VERSION_FILE_VERSION" != "$PACKAGE_JSON_VERSION" ]; then
  echo "❌ Error: VERSION file ($VERSION_FILE_VERSION) does not match package.json version ($PACKAGE_JSON_VERSION)"
  EXIT_CODE=1
fi

if [ $EXIT_CODE -eq 0 ]; then
  echo "✅ All versions are in sync!"
else
  echo "❌ Version mismatch detected. Please update all files to use the same version."
fi

exit $EXIT_CODE
