import { execSync } from 'child_process';
import { mkdtempSync, rmSync, writeFileSync } from 'fs';
import { join } from 'path';
import { tmpdir } from 'os';

export class GitTestHarness {
  tempDir: string;
  repoPath: string;

  constructor() {
    this.tempDir = mkdtempSync(join(tmpdir(), 'gitalisk-test-'));
    this.repoPath = join(this.tempDir, 'test-repo');
  }

  createRepository(): string {
    execSync(`git init "${this.repoPath}"`, { stdio: 'pipe' });
    execSync(`git config user.name "Test User"`, { cwd: this.repoPath, stdio: 'pipe' });
    execSync(`git config user.email "test@example.com"`, { cwd: this.repoPath, stdio: 'pipe' });
    if (process.platform === 'win32') {
      execSync('git config core.longpaths true', { cwd: this.repoPath, stdio: 'pipe' });
    }
    return this.repoPath;
  }

  createRepositoryWithBranch(branchName: string): string {
    this.createRepository();
    this.createInitialCommit();
    
    const currentBranch = execSync('git branch --show-current', { 
      cwd: this.repoPath, 
      encoding: 'utf8',
      stdio: 'pipe'
    }).trim();
    
    if (currentBranch === branchName) {
      return this.repoPath;
    }
    
    try {
      execSync(`git checkout -b "${branchName}"`, { cwd: this.repoPath, stdio: 'pipe' });
    } catch (error) {
      execSync(`git checkout "${branchName}"`, { cwd: this.repoPath, stdio: 'pipe' });
    }
    return this.repoPath;
  }

  createRepositoryWithDetachedHead(): string {
    this.createRepository();
    this.createInitialCommit();
    
    const commitHash = execSync('git rev-parse HEAD', { 
      cwd: this.repoPath, 
      encoding: 'utf8',
      stdio: 'pipe'
    }).trim();
    
    execSync(`git checkout "${commitHash}"`, { cwd: this.repoPath, stdio: 'pipe' });
    return this.repoPath;
  }

  createEmptyRepository(): string {
    this.createRepository();
    return this.repoPath;
  }

  createRepositoryWithSpecialCharacters(): string {
    this.createRepository();
    this.createInitialCommit();
    
    const specialBranchName = 'feature/test-123_special.branch';
    execSync(`git checkout -b "${specialBranchName}"`, { cwd: this.repoPath, stdio: 'pipe' });
    return this.repoPath;
  }

  createRepositoryWithUnicodeCharacters(): string {
    this.createRepository();
    this.createInitialCommit();
    
    const unicodeBranchName = 'feature/тест-ветка-🚀';
    execSync(`git checkout -b "${unicodeBranchName}"`, { cwd: this.repoPath, stdio: 'pipe' });
    return this.repoPath;
  }

  private createInitialCommit(): void {
    const testFile = join(this.repoPath, 'test.txt');
    writeFileSync(testFile, 'Initial test file\n');
    execSync('git add test.txt', { cwd: this.repoPath, stdio: 'pipe' });
    execSync('git commit -m "Initial commit"', { cwd: this.repoPath, stdio: 'pipe' });
  }

  addCommit(message: string = 'Test commit'): void {
    const fileName = `test-${Date.now()}.txt`;
    const testFile = join(this.repoPath, fileName);
    writeFileSync(testFile, `Test file created at ${new Date().toISOString()}\n`);
    execSync(`git add "${fileName}"`, { cwd: this.repoPath, stdio: 'pipe' });
    execSync(`git commit -m "${message}"`, { cwd: this.repoPath, stdio: 'pipe' });
  }

  addCommitWithAuthor(message: string, authorName: string, authorEmail: string): string {
    const fileName = `test-${Date.now()}.txt`;
    const testFile = join(this.repoPath, fileName);
    writeFileSync(testFile, `Test file created at ${new Date().toISOString()}\n`);
    execSync(`git add "${fileName}"`, { cwd: this.repoPath, stdio: 'pipe' });
    execSync(`git -c user.name="${authorName}" -c user.email="${authorEmail}" commit -m "${message}"`, { 
      cwd: this.repoPath, 
      stdio: 'pipe' 
    });
    
    // Return the commit hash
    return execSync('git rev-parse HEAD', { 
      cwd: this.repoPath, 
      encoding: 'utf8',
      stdio: 'pipe'
    }).trim();
  }

  createMultipleCommits(count: number, baseMessage: string = 'Commit'): string[] {
    const hashes: string[] = [];
    for (let i = 1; i <= count; i++) {
      this.addCommit(`${baseMessage} ${i}`);
      const hash = execSync('git rev-parse HEAD', { 
        cwd: this.repoPath, 
        encoding: 'utf8',
        stdio: 'pipe'
      }).trim();
      hashes.push(hash);
    }
    return hashes;
  }

  getCurrentCommitHash(): string {
    return execSync('git rev-parse HEAD', { 
      cwd: this.repoPath, 
      encoding: 'utf8',
      stdio: 'pipe'
    }).trim();
  }

  getPreviousCommitHash(stepsBack: number = 1): string {
    return execSync(`git rev-parse HEAD~${stepsBack}`, { 
      cwd: this.repoPath, 
      encoding: 'utf8',
      stdio: 'pipe'
    }).trim();
  }

  switchToBranch(branchName: string): void {
    try {
      execSync(`git checkout "${branchName}"`, { cwd: this.repoPath, stdio: 'pipe' });
    } catch (error) {
      execSync(`git checkout -b "${branchName}"`, { cwd: this.repoPath, stdio: 'pipe' });
    }
  }

  getCurrentBranchInRepo(): string {
    try {
      return execSync('git branch --show-current', { 
        cwd: this.repoPath, 
        encoding: 'utf8',
        stdio: 'pipe'
      }).trim();
    } catch (error) {
      return 'unknown';
    }
  }

  cleanup(): void {
    try {
      rmSync(this.tempDir, { recursive: true, force: true });
    } catch (error) {
      console.warn(`Failed to cleanup test directory: ${this.tempDir}`, error);
    }
  }
}

export function withTestHarness<T>(
  testFn: (harness: GitTestHarness) => T | Promise<T>
): () => Promise<T> {
  return async () => {
    const harness = new GitTestHarness();
    try {
      return await testFn(harness);
    } finally {
      harness.cleanup();
    }
  };
} 
