import { GitaliskWorkspaceFolder } from '@gitlab-org/gitalisk-node';
import path from 'node:path';

describe('gitalisk', () => {
  // Path to the repository root
  const workspacePath = path.join(process.cwd(), '..', '..' );

  test('getGitalisk should return repository status', async () => {
    // let's test that this is truly async and running in the libuv thread
    const gitaliskWorkspaceFolder = new GitaliskWorkspaceFolder({ 
      workspacePath,
      loggerOptions: {
        withColors: false,
        withStderr: false,
        withTarget: true,
        withThreadIds: true,
        withThreadNames: true,
      } 
    });
    
    const stats = await gitaliskWorkspaceFolder.indexAsync();
    expect(stats).toBeDefined();
    expect(stats.repoCount).toBeGreaterThan(0);
    expect(stats.fileCount).toBeGreaterThan(0);
    const repos = gitaliskWorkspaceFolder.getRepositories();
    expect(repos).toBeDefined();
    expect(repos.length).toBeGreaterThan(0);
    const firstRepo = repos[0];
    const files = firstRepo.getRepositoryFiles();
    expect(files).toBeDefined();
    expect(files.length).toBeGreaterThan(0);

    const inMemoryRepos = gitaliskWorkspaceFolder.getRepositories();
    expect(inMemoryRepos).toBeDefined();
    expect(inMemoryRepos.length).toBeGreaterThan(0);
    
    expect(gitaliskWorkspaceFolder.getWorkspacePath()).toBe(workspacePath);
    
    gitaliskWorkspaceFolder.cleanup();
    expect(gitaliskWorkspaceFolder.getRepositories().length).toBe(0);

  });

  test('indexAsync should be truly async and not block the event loop', async () => {
    const gitaliskWorkspaceFolder = new GitaliskWorkspaceFolder({ 
      workspacePath,
      loggerOptions: {
        withColors: false,
        withStderr: false,
        withTarget: true,
        withThreadIds: true,
        withThreadNames: true,
      } 
    });
    let flag = false;

    // Start the async operation but do not await it yet
    const promise = gitaliskWorkspaceFolder.indexAsync();

    // Synchronous code should run before the async operation completes
    flag = true;

    // Await the async operation
    await promise;

    // If the method was synchronous, flag would not be set before await
    expect(flag).toBe(true);

    // Additional checks as before
    const repos = gitaliskWorkspaceFolder.getRepositories();
    expect(repos).toBeDefined();
    expect(repos.length).toBeGreaterThan(0);
    gitaliskWorkspaceFolder.cleanup();
    expect(gitaliskWorkspaceFolder.getRepositories().length).toBe(0);
  });

  test('indexAsync does not block the event loop and allows other operations', async () => {
    const gitaliskWorkspaceFolder = new GitaliskWorkspaceFolder({ 
      workspacePath,
      loggerOptions: {
        withColors: false,
        withStderr: false,
        withTarget: true,
        withThreadIds: true,
        withThreadNames: true,
      } 
    });
    let otherOperationCompleted = false;

    // Start the async operation but do not await it yet
    const promise = gitaliskWorkspaceFolder.indexAsync();

    // Perform another operation (e.g., setTimeout)
    await new Promise(resolve => setTimeout(() => {
      otherOperationCompleted = true;
      resolve(undefined);
    }, 10)); // 10ms should be enough for a quick check

    // At this point, the other operation should have completed
    expect(otherOperationCompleted).toBe(true);

    // Now await the async operation
    await promise;

    // Usual checks
    const repos = gitaliskWorkspaceFolder.getRepositories();
    expect(repos).toBeDefined();
    expect(repos.length).toBeGreaterThan(0);
    gitaliskWorkspaceFolder.cleanup();
    expect(gitaliskWorkspaceFolder.getRepositories().length).toBe(0);
  });
}); 