import { GitaliskWorkspaceFolder } from "@gitlab-org/gitalisk-node";
import { GitTestHarness, withTestHarness } from "./test-harness";

describe("getCurrentBranch", () => {
  const createWorkspace = (workspacePath: string) => {
    return new GitaliskWorkspaceFolder({
      workspacePath,
      loggerOptions: {
        withColors: false,
        withStderr: false,
        withTarget: false,
        withThreadIds: false,
        withThreadNames: false,
      },
    });
  };

  describe("normal branch operations", () => {
    test(
      "should return current branch name for a regular branch",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepositoryWithBranch(
          "feature/awesome-feature"
        );
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const currentBranch = repo.getCurrentBranch();

        expect(currentBranch).toBe("feature/awesome-feature");

        workspace.cleanup();
      })
    );

    test(
      "should return main branch for default repository",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepository();
        harness.addCommit("Initial commit");

        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const currentBranch = repo.getCurrentBranch();

        // Git defaults to either 'main' or 'master' depending on configuration
        expect(currentBranch).toMatch(/^(main|master)$/);

        workspace.cleanup();
      })
    );

    test(
      "should handle master branch correctly",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepositoryWithBranch("master");

        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const currentBranch = repo.getCurrentBranch();

        expect(currentBranch).toBe("master");

        workspace.cleanup();
      })
    );
  });

  describe("special branch names", () => {
    test(
      "should handle branch names with special characters",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepositoryWithSpecialCharacters();
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const currentBranch = repo.getCurrentBranch();

        expect(currentBranch).toBe("feature/test-123_special.branch");

        workspace.cleanup();
      })
    );

    test(
      "should handle branch names with unicode characters",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepositoryWithUnicodeCharacters();
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const currentBranch = repo.getCurrentBranch();

        expect(currentBranch).toBe("feature/тест-ветка-🚀");

        workspace.cleanup();
      })
    );

    test(
      "should handle long branch names",
      withTestHarness(async (harness: GitTestHarness) => {
        const longBranchName =
          "feature/" + "very-long-branch-name-".repeat(10) + "end";
        const repoPath = harness.createRepositoryWithBranch(longBranchName);
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const currentBranch = repo.getCurrentBranch();

        expect(currentBranch).toBe(longBranchName);

        workspace.cleanup();
      })
    );
  });

  describe("detached HEAD state", () => {
    test(
      "should return commit hash for detached HEAD",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepositoryWithDetachedHead();
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const currentBranch = repo.getCurrentBranch();

        // Rust implementation returns "HEAD detached at {short_hash}" format
        expect(currentBranch).toMatch(/^HEAD detached at [a-f0-9]{7}$/);

        workspace.cleanup();
      })
    );
  });

  describe("empty repository handling", () => {
    test(
      "should handle empty repository gracefully",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createEmptyRepository();
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];

        expect(() => {
          const currentBranch = repo.getCurrentBranch();
        }).not.toThrow();

        workspace.cleanup();
      })
    );
  });

  describe("branch switching", () => {
    test(
      "should reflect current branch after switching",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepositoryWithBranch("initial-branch");
        let workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        let repositories = workspace.getRepositories();
        expect(repositories[0].getCurrentBranch()).toBe("initial-branch");
        workspace.cleanup();

        harness.switchToBranch("new-feature-branch");

        workspace = createWorkspace(harness.tempDir);
        await workspace.indexAsync();
        repositories = workspace.getRepositories();

        expect(repositories[0].getCurrentBranch()).toBe("new-feature-branch");

        workspace.cleanup();
      })
    );
  });

  describe("multiple repositories", () => {
    test(
      "should handle multiple repositories with different branches",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath1 = harness.createRepositoryWithBranch("repo1-branch");

        const harness2 = new GitTestHarness();
        const repoPath2 = harness2.createRepositoryWithBranch("repo2-branch");

        try {
          const workspace = createWorkspace(harness.tempDir);

          await workspace.indexAsync();
          const repositories = workspace.getRepositories();

          expect(repositories).toHaveLength(1);
          const repo1 = repositories.find((r) => r.path.includes("test-repo"));
          expect(repo1).toBeDefined();
          expect(repo1!.getCurrentBranch()).toBe("repo1-branch");

          workspace.cleanup();
        } finally {
          harness2.cleanup();
        }
      })
    );
  });

  describe("edge cases", () => {
    test(
      "should handle repository with unusual branch names",
      withTestHarness(async (harness: GitTestHarness) => {
        const branchName = "feat/JIRA-123/fix-issue-with-spaces";
        const repoPath = harness.createRepositoryWithBranch(branchName);
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const currentBranch = repo.getCurrentBranch();

        expect(currentBranch).toBe(branchName);

        workspace.cleanup();
      })
    );

    test(
      "should handle numeric branch names",
      withTestHarness(async (harness: GitTestHarness) => {
        const branchName = "123-feature";
        const repoPath = harness.createRepositoryWithBranch(branchName);
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const currentBranch = repo.getCurrentBranch();

        expect(currentBranch).toBe(branchName);

        workspace.cleanup();
      })
    );
  });
});
