import { GitaliskWorkspaceFolder } from "@gitlab-org/gitalisk-node";
import { GitTestHarness, withTestHarness } from "./test-harness";

describe("commit operations", () => {
  const createWorkspace = (workspacePath: string) => {
    return new GitaliskWorkspaceFolder({
      workspacePath,
      loggerOptions: {
        withColors: false,
        withStderr: false,
        withTarget: false,
        withThreadIds: false,
        withThreadNames: false,
      },
    });
  };

  describe("getCurrentCommitHash", () => {
    test(
      "should return current commit hash",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepository();
        harness.addCommit("Initial commit");
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const commitHash = repo.getCurrentCommitHash();

        expect(commitHash).toBeDefined();
        expect(commitHash).toHaveLength(40); // Full SHA-1 hash
        expect(commitHash).toMatch(/^[a-f0-9]{40}$/); // Hexadecimal pattern

        // Verify it matches what git actually returns
        const expectedHash = harness.getCurrentCommitHash();
        expect(commitHash).toBe(expectedHash);

        workspace.cleanup();
      })
    );

    test(
      "should handle repository with no commits gracefully",
      withTestHarness(async (harness: GitTestHarness) => {
        harness.createRepository();
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];

        expect(() => {
          repo.getCurrentCommitHash();
        }).toThrow();

        workspace.cleanup();
      })
    );
  });

  describe("getCommitsInfo", () => {
    test(
      "should get info for a single commit",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepository();
        harness.addCommit("Initial commit");
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const commitHash = repo.getCurrentCommitHash();
        const commitsInfo = repo.getCommitsInfo([commitHash]);

        expect(commitsInfo).toHaveLength(1);
        const commitInfo = commitsInfo[0];

        expect(commitInfo.hash).toBe(commitHash);
        expect(commitInfo.message).toBe("Initial commit");
        expect(commitInfo.authorName).toBe("Test User");
        expect(commitInfo.authorEmail).toBe("test@example.com");

        workspace.cleanup();
      })
    );

    test(
      "should get info for multiple commits",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepository();
        
        const commitHashes = harness.createMultipleCommits(5, "Test commit");
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];

        const commitsInfo = repo.getCommitsInfo(commitHashes);

        expect(commitsInfo).toHaveLength(5);

        for (let i = 0; i < commitsInfo.length; i++) {
          const commitInfo = commitsInfo[i];
          expect(commitInfo.hash).toBe(commitHashes[i]);
          expect(commitInfo.message).toBe(`Test commit ${i + 1}`);
          expect(commitInfo.authorName).toBe("Test User");
          expect(commitInfo.authorEmail).toBe("test@example.com");
        }

        workspace.cleanup();
      })
    );

    test(
      "should handle empty commit hash array",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepository();
        harness.addCommit("Initial commit");
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const commitsInfo = repo.getCommitsInfo([]);

        expect(commitsInfo).toHaveLength(0);

        workspace.cleanup();
      })
    );

    test(
      "should handle invalid commit hashes",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepository();
        harness.addCommit("Initial commit");
        const workspace = createWorkspace(harness.tempDir);

        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];

        expect(() => {
          repo.getCommitsInfo(["invalid_hash_123"]);
        }).toThrow();

        workspace.cleanup();
      })
    );

    test(
      "should handle commits with special characters in messages",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepository();
        
        const specialMessages = [
          "feat: add unicode support éñ español 🚀",
          "fix: handle pipe | character in commit messages",
          "docs: update README with <special> & symbols",
          "chore: refactor code with newlines", // Git %s format only shows subject line
        ];

        const commitHashes: string[] = [];
        for (const message of specialMessages) {
          harness.addCommit(message);
          commitHashes.push(harness.getCurrentCommitHash());
        }

        const workspace = createWorkspace(harness.tempDir);
        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const commitsInfo = repo.getCommitsInfo(commitHashes);

        expect(commitsInfo).toHaveLength(4);

        for (let i = 0; i < commitsInfo.length; i++) {
          const commitInfo = commitsInfo[i];
          expect(commitInfo.hash).toBe(commitHashes[i]);
          expect(commitInfo.message).toBe(specialMessages[i]);
        }

        workspace.cleanup();
      })
    );

    test(
      "should handle commits with different authors",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepository();
        
        const authors = [
          { name: "Alice Developer", email: "alice@example.com", message: "Alice's commit" },
          { name: "Bob Contributor", email: "bob@example.com", message: "Bob's commit" },
          { name: "Charlie Reviewer", email: "charlie@example.com", message: "Charlie's commit" },
        ];

        const commitHashes: string[] = [];
        for (const author of authors) {
          const hash = harness.addCommitWithAuthor(author.message, author.name, author.email);
          commitHashes.push(hash);
        }

        const workspace = createWorkspace(harness.tempDir);
        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];
        const commitsInfo = repo.getCommitsInfo(commitHashes);

        expect(commitsInfo).toHaveLength(3);

        for (let i = 0; i < commitsInfo.length; i++) {
          const commitInfo = commitsInfo[i];
          const expectedAuthor = authors[i];
          
          expect(commitInfo.hash).toBe(commitHashes[i]);
          expect(commitInfo.message).toBe(expectedAuthor.message);
          expect(commitInfo.authorName).toBe(expectedAuthor.name);
          expect(commitInfo.authorEmail).toBe(expectedAuthor.email);
        }

        workspace.cleanup();
      })
    );

    test(
      "should handle mixed valid and historical commits",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepository();
        
        harness.addCommit("First commit");
        const firstHash = harness.getCurrentCommitHash();
        
        harness.addCommit("Second commit");
        const secondHash = harness.getCurrentCommitHash();
        
        harness.addCommit("Third commit");
        const thirdHash = harness.getCurrentCommitHash();

        const workspace = createWorkspace(harness.tempDir);
        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];

        // Get info for non-consecutive commits
        const commitsInfo = repo.getCommitsInfo([thirdHash, firstHash, secondHash]);

        expect(commitsInfo).toHaveLength(3);

        // Verify order matches input order
        expect(commitsInfo[0].hash).toBe(thirdHash);
        expect(commitsInfo[0].message).toBe("Third commit");

        expect(commitsInfo[1].hash).toBe(firstHash);
        expect(commitsInfo[1].message).toBe("First commit");

        expect(commitsInfo[2].hash).toBe(secondHash);
        expect(commitsInfo[2].message).toBe("Second commit");

        workspace.cleanup();
      })
    );
  });

  describe("integration with other git operations", () => {
    test(
      "should work correctly with branch operations",
      withTestHarness(async (harness: GitTestHarness) => {
        const repoPath = harness.createRepositoryWithBranch("feature/test-commits");
        
        harness.addCommit("Feature commit 1");
        const featureHash1 = harness.getCurrentCommitHash();
        
        harness.addCommit("Feature commit 2");
        const featureHash2 = harness.getCurrentCommitHash();

        const workspace = createWorkspace(harness.tempDir);
        await workspace.indexAsync();
        const repositories = workspace.getRepositories();

        expect(repositories).toHaveLength(1);
        const repo = repositories[0];

        expect(repo.getCurrentBranch()).toBe("feature/test-commits");

        const commitsInfo = repo.getCommitsInfo([featureHash1, featureHash2]);

        expect(commitsInfo).toHaveLength(2);
        expect(commitsInfo[0].message).toBe("Feature commit 1");
        expect(commitsInfo[1].message).toBe("Feature commit 2");

        workspace.cleanup();
      })
    );
  });
}); 
