#![deny(clippy::all)]

use gitalisk_core::tracer::init_tracing;
use gitalisk_core::{
    repository::git::commit::CommitInfo as CoreCommitInfo,
    repository::gitalisk_repository::IterFileOptions, CoreGitaliskRepository,
    CoreGitaliskWorkspaceFolder,
};
use napi::bindgen_prelude::AbortSignal;
use napi::bindgen_prelude::AsyncTask;
use napi::bindgen_prelude::*;
use napi_derive::napi;
use std::sync::Arc;

#[napi]
pub struct GitaliskWorkspaceFolder {
    core_service: Arc<CoreGitaliskWorkspaceFolder>,
}

#[napi]
pub struct GitaliskRepository {
    inner: CoreGitaliskRepository,
}

impl From<CoreGitaliskRepository> for GitaliskRepository {
    fn from(core_repo: CoreGitaliskRepository) -> Self {
        Self { inner: core_repo }
    }
}

// Async task for indexing repositories
pub struct IndexRepositoriesTask {
    core_service: Arc<CoreGitaliskWorkspaceFolder>,
}

impl Task for IndexRepositoriesTask {
    type Output = WorkspaceStatistics;
    type JsValue = WorkspaceStatistics;

    fn compute(&mut self) -> napi::Result<Self::Output> {
        match self.core_service.index_repositories() {
            Ok(stats) => Ok(to_napi_workspace_statistics(&stats)),
            Err(e) => Err(napi::Error::from_reason(e)),
        }
    }

    fn resolve(&mut self, _env: Env, output: Self::Output) -> napi::Result<Self::JsValue> {
        Ok(output)
    }
}

#[napi(object)]
pub struct LoggerOptions {
    // Whether to use stderr for logging
    pub with_stderr: bool,
    // Whether to log colors
    pub with_colors: bool,
    // Whether to log the target
    pub with_target: bool,
    // Whether to log the thread ids
    pub with_thread_ids: bool,
    // Whether to log the thread names
    pub with_thread_names: bool,
}

#[napi(object)]
pub struct GitaliskWorkspaceOptions {
    // Workspace path to index
    pub workspace_path: String,
    // Logger options
    pub logger_options: LoggerOptions,
}

#[napi(object)]
#[derive(Clone)]
pub struct WorkspaceStatistics {
    pub file_count: u32,
    pub repo_count: u32,
}

fn to_napi_workspace_statistics(
    core: &gitalisk_core::workspace_folder::WorkspaceFolderStatistics,
) -> WorkspaceStatistics {
    WorkspaceStatistics {
        file_count: core.file_count,
        repo_count: core.repo_count,
    }
}

#[napi]
impl GitaliskWorkspaceFolder {
    #[napi(constructor)]
    pub fn new(options: GitaliskWorkspaceOptions) -> Self {
        init_tracing(
            options.logger_options.with_stderr,
            options.logger_options.with_colors,
            options.logger_options.with_target,
            options.logger_options.with_thread_ids,
            options.logger_options.with_thread_names,
        );

        GitaliskWorkspaceFolder {
            core_service: Arc::new(CoreGitaliskWorkspaceFolder::new(options.workspace_path)),
        }
    }

    /// Async index method that returns workspace statistics
    #[napi(ts_return_type = "Promise<WorkspaceStatistics>")]
    pub fn index_async(&self, signal: Option<AbortSignal>) -> AsyncTask<IndexRepositoriesTask> {
        AsyncTask::with_optional_signal(
            IndexRepositoriesTask {
                core_service: Arc::clone(&self.core_service),
            },
            signal,
        )
    }

    #[napi]
    pub fn get_workspace_path(&self) -> napi::Result<String> {
        Ok(self.core_service.get_workspace_path().to_string())
    }

    /// Get the currently indexed repositories from memory for the workspace
    #[napi]
    pub fn get_repositories(&self) -> napi::Result<Vec<GitaliskRepository>> {
        let core_repos = self.core_service.get_repositories();
        let repo_objects = core_repos
            .into_iter()
            .map(GitaliskRepository::from)
            .collect();
        Ok(repo_objects)
    }

    /// Cleanup method to clear all repositories from memory
    #[napi]
    pub fn cleanup(&self) -> napi::Result<()> {
        self.core_service.cleanup();
        Ok(())
    }
}

#[napi(object)]
pub struct CommitInfo {
    pub hash: String,
    pub message: String,
    #[napi(js_name = "authorName")]
    pub author_name: Option<String>,
    #[napi(js_name = "authorEmail")]
    pub author_email: Option<String>,
}

impl From<CoreCommitInfo> for CommitInfo {
    fn from(core: CoreCommitInfo) -> Self {
        Self {
            hash: core.hash,
            message: core.message,
            author_name: core.author_name,
            author_email: core.author_email,
        }
    }
}

#[napi(object)]
pub struct CommitAuthor {
    pub name: Option<String>,
    pub email: Option<String>,
}

#[napi(object)]
pub struct FileInfo {
    pub path: String,
    pub extension: String,
    pub name: String,
}

#[napi(object)]
pub struct FileIterOptions {
    /// Whether to include ignored files
    pub include_ignored: Option<bool>,
    /// Whether to include hidden files
    pub include_hidden: Option<bool>,
}

#[napi]
impl GitaliskRepository {
    #[napi(getter)]
    pub fn path(&self) -> String {
        self.inner.path.clone()
    }

    #[napi(getter)]
    pub fn workspace_path(&self) -> String {
        self.inner.workspace_path.clone()
    }

    #[napi]
    pub fn get_current_branch(&self) -> Result<String> {
        self.inner
            .get_current_branch()
            .map_err(|e| napi::Error::from_reason(e.to_string()))
    }

    #[napi]
    pub fn get_current_commit_hash(&self) -> Result<String> {
        self.inner
            .get_current_commit_hash()
            .map_err(|e| napi::Error::from_reason(e.to_string()))
    }

    #[napi]
    pub fn get_commits_info(&self, commit_hashes: Vec<String>) -> Result<Vec<CommitInfo>> {
        let hash_refs: Vec<&str> = commit_hashes.iter().map(|s| s.as_str()).collect();
        let core_commits = self
            .inner
            .get_commits_info(&hash_refs)
            .map_err(|e| napi::Error::from_reason(e.to_string()))?;

        Ok(core_commits.into_iter().map(CommitInfo::from).collect())
    }

    #[napi]
    pub fn list_branches(&self) -> Result<Vec<String>> {
        self.inner
            .list_branches()
            .map_err(|e| napi::Error::from_reason(e.to_string()))
    }

    #[napi]
    pub fn get_repository_files(&self, options: Option<FileIterOptions>) -> Result<Vec<FileInfo>> {
        let iter_options = match options {
            Some(opts) => IterFileOptions {
                include_ignored: opts.include_ignored.unwrap_or(false),
                include_hidden: opts.include_hidden.unwrap_or(false),
            },
            None => IterFileOptions {
                include_ignored: false,
                include_hidden: false,
            },
        };

        let file_info = self
            .inner
            .get_repo_files(iter_options)
            .map_err(|e| napi::Error::from_reason(e.to_string()))?;
        Ok(file_info
            .into_iter()
            .map(|f| FileInfo {
                path: f.path().to_string_lossy().to_string(),
                extension: f.extension().to_string(),
                name: f.name().to_string(),
            })
            .collect())
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use std::fs;
    use std::path::Path;
    use tempfile::tempdir;

    fn create_test_workspace_with_repos(
        workspace_path: &Path,
        repo_names: &[&str],
    ) -> std::io::Result<()> {
        for repo_name in repo_names {
            let repo_path = workspace_path.join(repo_name);
            fs::create_dir_all(&repo_path)?;

            // Create .git directory with config
            let git_dir = repo_path.join(".git");
            fs::create_dir_all(&git_dir)?;

            let config_content = r#"[core]
	repositoryformatversion = 0
	filemode = true
	bare = false
	logallrefupdates = true
"#;
            fs::write(git_dir.join("config"), config_content)?;

            // Create some sample files
            fs::write(repo_path.join("README.md"), format!("# {repo_name}"))?;
            fs::create_dir_all(repo_path.join("src"))?;
            fs::write(repo_path.join("src").join("main.rs"), "fn main() {}")?;
        }

        Ok(())
    }

    #[test]
    fn test_gitalisk_repository_from_core() {
        let core_repo =
            CoreGitaliskRepository::new("/test/repo".to_string(), "/test/workspace".to_string());

        let node_repo = GitaliskRepository::from(core_repo);
        assert_eq!(node_repo.path(), "/test/repo");
        assert_eq!(node_repo.workspace_path(), "/test/workspace");
    }

    #[test]
    fn test_gitalisk_repository_find_files() {
        let temp_dir = tempdir().unwrap();
        let repo_path = temp_dir.path();

        // Create some test files
        fs::create_dir_all(repo_path.join("src")).unwrap();
        fs::write(repo_path.join("README.md"), "test").unwrap();
        fs::write(repo_path.join("Cargo.toml"), "test").unwrap();
        fs::write(repo_path.join("src/main.rs"), "test").unwrap();

        let core_repo = CoreGitaliskRepository::new(
            repo_path.to_str().unwrap().to_string(),
            "/test/workspace".to_string(),
        );

        let node_repo = GitaliskRepository::from(core_repo);
        let files = node_repo.get_repository_files(None).unwrap();

        assert!(files.len() >= 3);

        // Check that expected files are found
        let file_names: Vec<String> = files
            .iter()
            .filter_map(|f| Path::new(&f.path).file_name()?.to_str())
            .map(String::from)
            .collect();

        assert!(file_names.contains(&"README.md".to_string()));
        assert!(file_names.contains(&"Cargo.toml".to_string()));
        assert!(file_names.contains(&"main.rs".to_string()));
    }

    #[test]
    fn test_index_repositories_task() {
        let temp_dir = tempdir().unwrap();
        let workspace_path = temp_dir.path();

        // Create test repositories
        create_test_workspace_with_repos(workspace_path, &["repo1", "repo2"]).unwrap();

        let core_service = Arc::new(CoreGitaliskWorkspaceFolder::new(
            workspace_path.to_str().unwrap().to_string(),
        ));
        let mut task = IndexRepositoriesTask { core_service };

        let result = task.compute().unwrap();

        // Since we created 2 repos, we should have at least some files
        if result.repo_count > 0 {
            assert!(result.file_count > 0);
        }
    }

    #[test]
    fn test_to_napi_workspace_statistics() {
        let core_stats = gitalisk_core::workspace_folder::WorkspaceFolderStatistics {
            file_count: 42,
            repo_count: 7,
        };

        let napi_stats = to_napi_workspace_statistics(&core_stats);
        assert_eq!(napi_stats.file_count, 42);
        assert_eq!(napi_stats.repo_count, 7);
    }

    #[test]
    fn test_workspace_statistics_clone() {
        let stats = WorkspaceStatistics {
            file_count: 15,
            repo_count: 3,
        };

        let cloned = stats.clone();
        assert_eq!(cloned.file_count, 15);
        assert_eq!(cloned.repo_count, 3);
    }

    #[test]
    fn test_logger_options_struct() {
        let logger_opts = LoggerOptions {
            with_stderr: true,
            with_colors: false,
            with_target: true,
            with_thread_ids: false,
            with_thread_names: true,
        };

        assert!(logger_opts.with_stderr);
        assert!(!logger_opts.with_colors);
        assert!(logger_opts.with_target);
        assert!(!logger_opts.with_thread_ids);
        assert!(logger_opts.with_thread_names);
    }

    #[test]
    fn test_gitalisk_workspace_options_struct() {
        let logger_opts = LoggerOptions {
            with_stderr: true,
            with_colors: true,
            with_target: false,
            with_thread_ids: true,
            with_thread_names: false,
        };

        let workspace_opts = GitaliskWorkspaceOptions {
            workspace_path: "/test/workspace".to_string(),
            logger_options: logger_opts,
        };

        assert!(workspace_opts.logger_options.with_stderr);
        assert!(workspace_opts.logger_options.with_colors);
        assert!(!workspace_opts.logger_options.with_target);
    }

    // Note: We can't easily test the NAPI functions directly in unit tests
    // since they require a Node.js runtime. Those tests should be covered by
    // packages/tests/src/gitalisk.test.ts
}
