use crate::repository::gitalisk_repository::CoreGitaliskRepository;
use ignore::WalkBuilder;
use std::collections::HashMap;
use std::sync::{Arc, Mutex};
use tracing::info;

#[derive(Clone, Debug)]
pub struct WorkspaceFolderStatistics {
    pub file_count: u32,
    pub repo_count: u32,
}

pub struct CoreGitaliskWorkspaceFolder {
    workspace_path: String,
    repositories: Arc<Mutex<HashMap<String, CoreGitaliskRepository>>>,
    statistics: Arc<Mutex<WorkspaceFolderStatistics>>,
}

impl CoreGitaliskWorkspaceFolder {
    pub fn new(workspace_path: String) -> Self {
        Self {
            workspace_path,
            repositories: Arc::new(Mutex::new(HashMap::new())),
            statistics: Arc::new(Mutex::new(WorkspaceFolderStatistics {
                file_count: 0,
                repo_count: 0,
            })),
        }
    }

    // re-index repositories in the workspace folder
    // this returns how many files were scanned and how many repositories were found
    // to access the repositories, use the `get_repositories` method
    pub fn index_repositories(&self) -> Result<WorkspaceFolderStatistics, String> {
        // clear existing repositories and statistics
        self.cleanup();

        info!(
            "Indexing repositories in workspace: {}",
            self.workspace_path
        );
        let repos = Arc::new(Mutex::new(Vec::new()));
        let repositories = Arc::clone(&self.repositories);
        let workspace_path = self.workspace_path.clone();
        let statistics = Arc::clone(&self.statistics);

        // note we explicit set the above filters to false to allow for discovering
        // various repositories within a workspace folder while still leveraging the
        // parallelism provided by ignore's walk builder.
        WalkBuilder::new(&workspace_path)
            // Enables ignoring hidden files.
            .hidden(false)
            // Enables reading `.gitignore` files.
            .git_ignore(false)
            // Enables reading a global gitignore file, whose path is specified in
            // git's `core.excludesFile` config option.
            .git_global(false)
            // Enables reading `.git/info/exclude` files.
            .git_exclude(false)
            // Enables reading ignore files from parent directories.
            .ignore(false)
            // Enables reading ignore files from parent directories.
            .parents(false)
            .build_parallel()
            .run(|| {
                let repos = Arc::clone(&repos);
                let repositories = Arc::clone(&repositories);
                let workspace_path = workspace_path.clone();
                let statistics = statistics.clone();

                Box::new(move |result| {
                    if let Ok(entry) = result {
                        // Handle file counting with minimal lock duration
                        if entry.file_type().map(|ft| ft.is_file()).unwrap_or(false) {
                            statistics.lock().unwrap().file_count += 1;
                        }

                        // Handle repository detection
                        if entry.file_type().map(|ft| ft.is_dir()).unwrap_or(false)
                            && entry.file_name() == ".git"
                        {
                            if let Some(parent) = entry.path().parent() {
                                if let Some(repo_path) = parent.to_str() {
                                    // Do file I/O without holding any locks
                                    let git_config_path = entry.path().join("config");
                                    if git_config_path.is_file() {
                                        // Create repository object without locks
                                        let repo_obj = CoreGitaliskRepository::new(
                                            repo_path.to_string(),
                                            workspace_path.clone(),
                                        );

                                        // Acquire locks only when needed, release quickly
                                        {
                                            let mut stats = statistics.lock().unwrap();
                                            stats.repo_count += 1;
                                        } // stats lock released here

                                        {
                                            let mut repos_map = repositories.lock().unwrap();
                                            repos_map
                                                .insert(repo_path.to_string(), repo_obj.clone());
                                        }

                                        {
                                            repos.lock().unwrap().push(repo_obj);
                                        }
                                    }
                                }
                            }
                        }
                    }
                    ignore::WalkState::Continue
                })
            });

        info!(
            "Finished indexing repositories in workspace: {}, statistics: {:?}",
            self.workspace_path,
            self.statistics.lock().unwrap()
        );

        Ok(self.statistics.lock().unwrap().clone())
    }

    pub fn get_repositories(&self) -> Vec<CoreGitaliskRepository> {
        let repos = self.repositories.lock().unwrap();
        repos.values().cloned().collect()
    }

    pub fn get_workspace_path(&self) -> &str {
        &self.workspace_path
    }

    pub fn cleanup(&self) {
        info!("Cleaning up workspace: {}", self.workspace_path);
        self.repositories.lock().unwrap().clear();
        let mut stats = self.statistics.lock().unwrap();
        stats.file_count = 0;
        stats.repo_count = 0;
    }
}
