# gitalisk-node

> **Note:** This project is a work in progress and not all functionality is available yet. TODO: remove the note when it's ready.

![Pipeline Status](https://gitlab.com/gitlab-org/rust/gitalisk/badges/main/pipeline.svg)

A high-performance Node.js native addon for querying Git repository status and metadata, powered by Rust. `gitalisk-node` provides fast, comprehensive access to Git workspace and repository information—such as file status, branches, and more—directly from Node.js, without needing the git CLI. It is ideal for tools that need to efficiently scan, index, or monitor many repositories.

This project is structured as a monorepo with Rust and Node.js workspaces:
- **Rust core (`gitalisk-core`)**: Implements the core logic for querying Git repositories.
- **Rust Node binding (`gitalisk-node`)**: Exposes the core logic to Node.js via NAPI, with async and workspace-level APIs.
- **Node.js package (`@gitlab-org/gitalisk-node`)**: The published npm module, with TypeScript definitions and a simple API.
- **Test suite (`packages/tests`)**: Comprehensive Jest tests for the Node.js API.

Key features:
- Fast, parallelized status checking using Rust
- No dependency on the git binary
- Async and sync APIs for workspace and repository operations
- TypeScript definitions included
- Designed for use in developer tools, editors, and CI systems

## Installation

```bash
npm install @gitlab-org/gitalisk-node
```

## Features

- Fast git status checking using Rust
- No git binary dependency
- Simple API
- TypeScript definitions included

## Usage

```typescript
import { GitaliskWorkspaceFolder, LoggerOptions } from '@gitlab-org/gitalisk-node';

// Configure logging options (all fields are optional)
const loggerOptions: LoggerOptions = {
  withStderr: false,
  withColors: true,
  withTarget: false,
  withThreadIds: false,
  withThreadNames: false,
};

const workspacePath = '/path/to/your/projects';

// Create a workspace service for a specific workspace path
const workspace = new GitaliskWorkspaceFolder({ 
  workspacePath,
  loggerOptions 
});

// Index all repositories in the workspace directory (async)
await workspace.indexAsync();

// Get the workspace path
console.log('Workspace path:', workspace.getWorkspacePath());

// Get all indexed repositories
const repos = workspace.getRepositories();
for (const repo of repos) {
  console.log('Repo path:', repo.path);
  console.log('Current branch:', repo.getCurrentBranch());
  console.log('Branches:', repo.listBranches());
  console.log('Repo files:', repo.findRepoFiles());
}

// Get workspace statistics
const stats = workspace.getWorkspaceStatistics();
console.log('Workspace stats:', stats);

// Cleanup (remove all indexed repositories from memory)
workspace.cleanup();
```

- `GitaliskWorkspaceFolder` is the main entry point for indexing and querying multiple repositories in a directory tree. Each workspace instance is focused on a single workspace path.
- All APIs are available in TypeScript.
- `indexAsync` is non-blocking and can be aborted with an `AbortSignal`.
- You can call `getRepositories` and `getWorkspaceStatistics` at any time after indexing.
- `GitaliskRepository` exposes methods to query branch info and file lists for each repository.

## Project Structure

This project is set up as an npm workspace monorepo:

```
gitalisk/
├── package.json (workspace root)
├── crates/
│   └── gitalisk-node/ (Rust NAPI source)
└── packages/
    ├── gitalisk-node/ (Node.js addon package)
    └── tests/                   (Jest tests package)
```

## Development

### Prerequisites

Ensure you have the following installed:

- [Mise](https://mise.jdx.dev/) (for tool management)
- [Lefthook](https://github.com/evilmartians/lefthook) (for managing git hooks)

We use **mise** to manage the proper version of runtimes and tools. 

### Setting up Lefthook

1. Install Lefthook globally:

   ```bash
   # On macOS
   brew install lefthook

   # Other platforms
   # See https://lefthook.dev/installation/
   ```

2. Enable Lefthook in the project:

   ```bash
   lefthook install
   ```

This will set up git hooks as defined in the `lefthook.yml` file.

### Setting up Mise

1. [Install mise](https://mise.jdx.dev/getting-started.html#installing-mise-cli) for your system.

2. Activate mise:

    ```bash
    mise activate
    ```

    For Windows make sure you have PowerShell 7 [installed](https://learn.microsoft.com/en-us/powershell/scripting/install/installing-powershell-on-windows) and running, then you activate mise with:

    ```bash
    mise activate pwsh
    ```

3. Optionaly add [automatic mise activation](https://mise.jdx.dev/installing-mise.html#shells) to your shell profile.

4. Verify mise activation went successful by opening a project folder and running:

    ```bash
    # Install all required tools using mise
    mise install
    ```

    ```bash
    # List all tools and their versions used in the project
    mise ls
    ```

    ```bash
    # Verify mise activation properly updated PATH with the correct tool version
    npm -v
    ```

### Setting up the Workspace

```bash
# Install all dependencies across the workspace
npm install
```

### Building

```bash
# Build the NAPI module
npm run build
```

### Testing

```bash
# Run Node.js tests (delegates to the tests package)
npm test

# Run tests in watch mode
npm run test:watch

# Run Rust tests
cargo test --workspace

# Run Rust tests with verbose output
cargo test --workspace --verbose
```

### Linting and Formatting

```bash
# Check Rust code with Clippy
cargo clippy --workspace -- -D warnings

# Format Rust code
cargo fmt --all -- --check

# Verify Rust code compiles
cargo check --workspace
```

### Publishing

To publish the package to npm:

```bash
# From the workspace root
npm publish -w @gitlab-org/gitalisk-node

# Or directly from the package directory
cd packages/gitalisk-node
npm publish
```

## License

MIT OR Apache-2.0
