pub mod test_helpers;

use crate::test_helpers::*;
use glql::{compile, utils::feature_flags::FeatureFlag};
use serde_json::Value as JsonValue;

#[test]
fn test_type_equals_merge_request() {
    // all MRs - edge case when no filters are provided
    assert_eq!(
        serde_json::from_str::<JsonValue>(
            compile(
                "type = MergeRequest",
                r#"{"fields": "id, title", "project": "gitlab-org/gitlab"}"#
            )
            .as_str()
        )
        .unwrap()["output"]
            .as_str()
            .unwrap()
            .lines()
            .nth(2)
            .unwrap(),
        "    mergeRequests(before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_with_project_context(
            "gitlab-org/gitlab",
            "type = MergeRequest and state = merged"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    mergeRequests(state: merged, before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_with_project_context(
            "gitlab-org/gitlab",
            "reviewer = @foo and type = MergeRequest and state = merged"
        )
        .lines()
        .nth(2)
        .unwrap(),
        "    mergeRequests(reviewerUsername: \"foo\", state: merged, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_type_equals_issue() {
    assert_eq!(
        compile_graphql("type = Issue and state = opened")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(types: ISSUE, state: opened, before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql("confidential = true and type = Issue")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(confidential: true, types: ISSUE, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_type_equals_work_item_type() {
    assert_eq!(
        compile_graphql("type = Objective and state = opened")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(types: OBJECTIVE, state: opened, before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql("type = KeyResult and state = opened")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(types: KEY_RESULT, state: opened, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_type_equals_invalid() {
    assert_eq!(
        compile_graphql("state = opened and type = none"),
        "Error: `type` cannot be compared with `NONE`. Supported value types: `Enum` (`Issue`, `Incident`, `TestCase`, `Requirement`, `Task`, `Ticket`, `Objective`, `KeyResult`, `Epic`, `MergeRequest`), `List`."
    );

    assert_eq!(
        compile_graphql("label = \"fish\" and type = none"),
        "Error: `type` cannot be compared with `NONE`. Supported value types: `Enum` (`Issue`, `Incident`, `TestCase`, `Requirement`, `Task`, `Ticket`, `Objective`, `KeyResult`, `Epic`, `MergeRequest`), `List`."
    );
}

#[test]
fn test_type_equals_valid_but_invalid_fields() {
    assert_eq!(
        compile_graphql("reviewer = @foo and type = Issue"),
        "Error: `reviewer` is not a recognized field for work items."
    );

    assert_eq!(
        compile_graphql("confidential = true and type = MergeRequest"),
        "Error: `confidential` is not a recognized field for merge requests."
    )
}

#[test]
fn type_test_in_work_item_types() {
    assert_eq!(
        compile_graphql("type in (Issue, Incident, TestCase)")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(types: [ISSUE, INCIDENT, TEST_CASE], before: $before, after: $after, first: $limit) {"
    );

    assert_eq!(
        compile_graphql("type in (Objective, KeyResult)")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(types: [OBJECTIVE, KEY_RESULT], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn type_test_in_work_item_types_with_epic() {
    // with FF enabled: allowed
    FeatureFlag::GlqlWorkItems.set(true);

    assert_eq!(
        compile_with_group_context("gitlab-org", "type in (Epic, Issue)")
            .lines()
            .nth(2)
            .unwrap(),
        "    workItems(types: [EPIC, ISSUE], before: $before, after: $after, first: $limit, includeDescendants: true) {"
    );

    // with FF disabled: not allowed
    FeatureFlag::GlqlWorkItems.set(false);
    assert_eq!(
        compile_with_group_context("gitlab-org", "type in (Epic, Issue)"),
        "Error: Type `Epic` cannot be combined with other types (`Issue`) in the same query. Try using `type = Epic` instead."
    );
}

#[test]
fn type_test_in_invalid_type() {
    assert_eq!(
        compile_graphql("type in (MergeRequest, Issue, Task)"),
        "Error: Type `MergeRequest` cannot be combined with other types (`Issue`, `Task`) in the same query. Try using `type = MergeRequest` instead."
    );

    assert_eq!(
        compile_graphql("type in (Epic, MergeRequest, Issue, Task, Objective, KeyResult)"),
        "Error: Type `Epic` cannot be combined with other types (`MergeRequest`, `Issue`, `Task`, `Objective`, `KeyResult`) in the same query. Try using `type = Epic` instead."
    );
}
