pub mod test_helpers;

use crate::test_helpers::*;
use glql::utils::feature_flags::FeatureFlag;
use serde_json::json;

fn mock_data() -> String {
    serde_json::to_string(&json!({
        "project": {
            "mergeRequests": {
                "nodes": [
                    {
                        "id": "gid://gitlab/MergeRequest/47",
                        "iid": "8",
                        "title": "Consequuntur ipsum sint dolores accusamus minus iste illo libero.",
                        "webUrl": "http://127.0.0.1:3000/gitlab-org/gitlab-shell/-/merge_requests/8",
                        "reference": "!8",
                        "state": "merged",
                        "updatedAt": "2025-03-11T12:45:26Z",
                        "createdAt": "2025-03-11T12:45:26Z",
                        "mergeUser": {
                            "id": "gid://gitlab/User/6",
                            "avatarUrl": r#"https://www.gravatar.com/avatar/62dfc262b7f2fb35201111e3f88fd8c9806f23620ef4232502567ec3fccf8e14?s=80\u0026d=identicon"#,
                            "username": "lidia_reynolds",
                            "name": "Dorris Kulas",
                            "webUrl": "http://127.0.0.1:3000/lidia_reynolds",
                            "__typename": "UserCore"
                        },
                        "labels": {
                            "nodes": [],
                            "__typename": "LabelConnection"
                        },
                        "lastComment": {
                            "nodes": [],
                            "__typename": "NoteConnection"
                        },
                        "__typename": "MergeRequest"
                    },
                    {
                        "id": "gid://gitlab/MergeRequest/19",
                        "iid": "7",
                        "title": "Cupiditate at sequi voluptatem aliquam quisquam magnam sit.",
                        "webUrl": "http://127.0.0.1:3000/gitlab-org/gitlab-shell/-/merge_requests/7",
                        "reference": "!7",
                        "state": "opened",
                        "updatedAt": "2025-04-11T11:30:47Z",
                        "createdAt": "2025-03-11T12:40:20Z",
                        "mergeUser": null,
                        "labels": {
                            "nodes": [
                                {
                                    "id": "gid://gitlab/GroupLabel/18",
                                    "title": "Afterfunc",
                                    "color": "#d74690",
                                    "textColor": "#FFFFFF",
                                    "__typename": "Label"
                                },
                                {
                                    "id": "gid://gitlab/GroupLabel/12",
                                    "title": "Afternix",
                                    "color": "#c0be6a",
                                    "textColor": "#1F1E24",
                                    "__typename": "Label"
                                },
                                {
                                    "id": "gid://gitlab/ProjectLabel/84",
                                    "title": "Corolla Wagon",
                                    "color": "#23d305",
                                    "textColor": "#1F1E24",
                                    "__typename": "Label"
                                },
                                {
                                    "id": "gid://gitlab/ProjectLabel/83",
                                    "title": "Durango",
                                    "color": "#8003a0",
                                    "textColor": "#FFFFFF",
                                    "__typename": "Label"
                                },
                                {
                                    "id": "gid://gitlab/ProjectLabel/85",
                                    "title": "Echo",
                                    "color": "#8335e4",
                                    "textColor": "#FFFFFF",
                                    "__typename": "Label"
                                },
                                {
                                    "id": "gid://gitlab/GroupLabel/11",
                                    "title": "TH",
                                    "color": "#3d8bd5",
                                    "textColor": "#FFFFFF",
                                    "__typename": "Label"
                                }
                            ],
                            "__typename": "LabelConnection"
                        },
                        "lastComment": {
                            "nodes": [
                                {
                                    "bodyHtml": r#"\u003cdiv class=\"gl-relative markdown-code-block js-markdown-code\"\u003e\n\u003cpre data-sourcepos=\"1:1-3:3\" data-canonical-lang=\"glql\" class=\"code highlight js-syntax-highlight language-plaintext\" v-pre=\"true\"\u003e\u003ccode\u003e\u003cspan id=\"LC1\" class=\"line\" lang=\"plaintext\"\u003eassignee = @root\u003c/span\u003e\u003c/code\u003e\u003c/pre\u003e\n\u003ccopy-code\u003e\u003c/copy-code\u003e\u003cinsert-code-snippet\u003e\u003c/insert-code-snippet\u003e\n\u003c/div\u003e"#,
                                    "__typename": "Note"
                                }
                            ],
                            "__typename": "NoteConnection"
                        },
                        "__typename": "MergeRequest"
                    }
                ],
                "pageInfo": {
                    "startCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wMy0xMSAxMjo0NToyNi4xMTMxOTMwMDAgKzAwMDAiLCJpZCI6IjQ3In0",
                    "endCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wMy0xMSAxMjo0MDoyMC4xMzUxNjUwMDAgKzAwMDAiLCJpZCI6IjE5In0",
                    "hasNextPage": false,
                    "hasPreviousPage": false,
                    "__typename": "PageInfo"
                },
                "count": 2,
                "__typename": "MergeRequestConnection"
            },
            "__typename": "Project"
        }
    })).unwrap()
}

#[test]
fn test_parse_real_world_data() {
    let data = mock_data();
    let output = transform_data(
        &data,
        r#"id, iid, state, title, updated, createdAt, updatedAt, mergedBy, labels("Aft*") as "Aft", labels as "Remaining Labels", lastComment as "The last comment!""#,
    );

    assert_eq!(
        output["fields"],
        json!([
            { "key": "id", "label": "ID", "name": "id" },
            { "key": "iid", "label": "IID", "name": "iid" },
            { "key": "state", "label": "State", "name": "state" },
            { "key": "title", "label": "Title", "name": "title" },
            { "key": "updated", "label": "Updated", "name": "updatedAt" },
            { "key": "createdAt", "label": "Created at", "name": "createdAt" },
            { "key": "updatedAt", "label": "Updated at", "name": "updatedAt" },
            { "key": "mergedBy", "label": "Merged by", "name": "mergeUser" },
            { "key": "labels_Aft*", "label": "Aft", "name": "labels" },
            { "key": "labels", "label": "Remaining Labels", "name": "labels" },
            { "key": "lastComment", "label": "The last comment!", "name": "lastComment" },
        ])
    );

    assert_eq!(
        output["data"],
        json!({
            "__typename": "MergeRequestConnection",
            "count": 2,
            "nodes": [{
                "__typename": "MergeRequest",
                "createdAt": "2025-03-11T12:45:26Z",
                "id": "gid://gitlab/MergeRequest/47",
                "iid": "8",
                "labels": {
                    "nodes": []
                },
                "labels_Aft*": {
                    "nodes": []
                },
                "lastComment": null,
                "mergeUser": {
                    "__typename": "UserCore",
                    "avatarUrl": "https://www.gravatar.com/avatar/62dfc262b7f2fb35201111e3f88fd8c9806f23620ef4232502567ec3fccf8e14?s=80\\u0026d=identicon",
                    "id": "gid://gitlab/User/6",
                    "name": "Dorris Kulas",
                    "username": "lidia_reynolds",
                    "webUrl": "http://127.0.0.1:3000/lidia_reynolds"
                },
                "mergedBy": {
                    "__typename": "UserCore",
                    "avatarUrl": "https://www.gravatar.com/avatar/62dfc262b7f2fb35201111e3f88fd8c9806f23620ef4232502567ec3fccf8e14?s=80\\u0026d=identicon",
                    "id": "gid://gitlab/User/6",
                    "name": "Dorris Kulas",
                    "username": "lidia_reynolds",
                    "webUrl": "http://127.0.0.1:3000/lidia_reynolds"
                },
                "reference": "!8",
                "state": "merged",
                "title": "Consequuntur ipsum sint dolores accusamus minus iste illo libero.",
                "updated": "2025-03-11T12:45:26Z",
                "updatedAt": "2025-03-11T12:45:26Z",
                "webUrl": "http://127.0.0.1:3000/gitlab-org/gitlab-shell/-/merge_requests/8"
            }, {
                "__typename": "MergeRequest",
                "createdAt": "2025-03-11T12:40:20Z",
                "id": "gid://gitlab/MergeRequest/19",
                "iid": "7",
                "labels": {
                    "nodes": [{
                        "__typename": "Label",
                        "color": "#23d305",
                        "id": "gid://gitlab/ProjectLabel/84",
                        "textColor": "#1F1E24",
                        "title": "Corolla Wagon"
                    }, {
                        "__typename": "Label",
                        "color": "#8003a0",
                        "id": "gid://gitlab/ProjectLabel/83",
                        "textColor": "#FFFFFF",
                        "title": "Durango"
                    }, {
                        "__typename": "Label",
                        "color": "#8335e4",
                        "id": "gid://gitlab/ProjectLabel/85",
                        "textColor": "#FFFFFF",
                        "title": "Echo"
                    }, {
                        "__typename": "Label",
                        "color": "#3d8bd5",
                        "id": "gid://gitlab/GroupLabel/11",
                        "textColor": "#FFFFFF",
                        "title": "TH"
                    }]
                },
                "labels_Aft*": {
                    "nodes": [{
                        "__typename": "Label",
                        "color": "#d74690",
                        "id": "gid://gitlab/GroupLabel/18",
                        "textColor": "#FFFFFF",
                        "title": "Afterfunc"
                    }, {
                        "__typename": "Label",
                        "color": "#c0be6a",
                        "id": "gid://gitlab/GroupLabel/12",
                        "textColor": "#1F1E24",
                        "title": "Afternix"
                    }]
                },
                "lastComment": "\\u003cdiv class=\\\"gl-relative markdown-code-block js-markdown-code\\\"\\u003e\\n\\u003cpre data-sourcepos=\\\"1:1-3:3\\\" data-canonical-lang=\\\"glql\\\" class=\\\"code highlight js-syntax-highlight language-plaintext\\\" v-pre=\\\"true\\\"\\u003e\\u003ccode\\u003e\\u003cspan id=\\\"LC1\\\" class=\\\"line\\\" lang=\\\"plaintext\\\"\\u003eassignee = @root\\u003c/span\\u003e\\u003c/code\\u003e\\u003c/pre\\u003e\\n\\u003ccopy-code\\u003e\\u003c/copy-code\\u003e\\u003cinsert-code-snippet\\u003e\\u003c/insert-code-snippet\\u003e\\n\\u003c/div\\u003e",
                "mergeUser": null,
                "mergedBy": null,
                "reference": "!7",
                "state": "opened",
                "title": "Cupiditate at sequi voluptatem aliquam quisquam magnam sit.",
                "updated": "2025-04-11T11:30:47Z",
                "updatedAt": "2025-04-11T11:30:47Z",
                "webUrl": "http://127.0.0.1:3000/gitlab-org/gitlab-shell/-/merge_requests/7"
            }],
          "pageInfo": {
                "__typename": "PageInfo",
                "endCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wMy0xMSAxMjo0MDoyMC4xMzUxNjUwMDAgKzAwMDAiLCJpZCI6IjE5In0",
                "hasNextPage": false,
                "hasPreviousPage": false,
                "startCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wMy0xMSAxMjo0NToyNi4xMTMxOTMwMDAgKzAwMDAiLCJpZCI6IjQ3In0"
          }
        })
    )
}

#[test]
fn test_transform_work_items() {
    let data = serde_json::to_string(&json!({
        "group": {
            "workItems": {
                "nodes": [{
                    "id": "gid://gitlab/WorkItem/618",
                    "iid": "3",
                    "title": "Consectetur nisi unde dolore cupiditate laborum et.",
                    "webUrl": "http://127.0.0.1:3000/groups/gitlab-org/-/epics/3",
                    "reference": "#3",
                    "state": "OPEN",
                    "updatedAt": "2025-08-04T09:32:02Z",
                    "widgets": [
                        { "__typename": "WorkItemWidgetAssignees" },
                        { "__typename": "WorkItemWidgetAwardEmoji" },
                        { "__typename": "WorkItemWidgetColor" },
                        { "__typename": "WorkItemWidgetCurrentUserTodos" },
                        { "__typename": "WorkItemWidgetCustomFields" },
                        { "__typename": "WorkItemWidgetDescription" },
                        { "__typename": "WorkItemWidgetHealthStatus" },
                        {
                            "type": "HIERARCHY",
                            "parent": {
                                "id": "gid://gitlab/WorkItem/620",
                                "iid": "5",
                                "reference": "#5",
                                "state": "CLOSED",
                                "title": "Aperiam quasi iure error reprehenderit quisquam totam dolore aliquid debitis fugiat.",
                                "webUrl": "http://127.0.0.1:3000/groups/gitlab-org/-/epics/5",
                                "__typename": "WorkItem"
                            },
                            "__typename": "WorkItemWidgetHierarchy"
                        },
                        { "__typename": "WorkItemWidgetLabels" },
                        { "__typename": "WorkItemWidgetLinkedItems" },
                        { "__typename": "WorkItemWidgetMilestone" },
                        { "__typename": "WorkItemWidgetNotes" },
                        { "__typename": "WorkItemWidgetNotifications" },
                        { "__typename": "WorkItemWidgetParticipants" },
                        { "__typename": "WorkItemWidgetStartAndDueDate" },
                        { "__typename": "WorkItemWidgetVerificationStatus" },
                        { "__typename": "WorkItemWidgetTimeTracking" },
                        { "__typename": "WorkItemWidgetWeight"}
                    ],
                    "__typename": "WorkItem"
                }],
                "pageInfo": {
                    "startCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wMy0wOCAxMjo0NToyMi4xNTgxMjcwMDAgKzAwMDAiLCJpZCI6IjYxOCJ9",
                    "endCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wMy0wOCAxMjo0NToyMi4xNTgxMjcwMDAgKzAwMDAiLCJpZCI6IjYxOCJ9",
                    "hasNextPage": false,
                    "hasPreviousPage": false,
                    "__typename": "PageInfo"
                },
                "count": 1,
                "__typename": "WorkItemConnection"
            },
            "__typename": "Group"
        }
    })).unwrap();

    let output = transform_data(&data, r#"title, updated, epic"#);

    assert_eq!(
        output["data"],
        json!({
            "__typename": "WorkItemConnection",
            "count": 1,
            "nodes": [
                {
                    "__typename": "WorkItem",
                    "epic": {
                        "__typename": "WorkItem",
                        "id": "gid://gitlab/WorkItem/620",
                        "iid": "5",
                        "reference": "#5",
                        "state": "CLOSED",
                        "title": "Aperiam quasi iure error reprehenderit quisquam totam dolore aliquid debitis fugiat.",
                        "webUrl": "http://127.0.0.1:3000/groups/gitlab-org/-/epics/5"
                    },
                    "id": "gid://gitlab/WorkItem/618",
                    "iid": "3",
                    "parent": {
                        "__typename": "WorkItem",
                        "id": "gid://gitlab/WorkItem/620",
                        "iid": "5",
                        "reference": "#5",
                        "state": "CLOSED",
                        "title": "Aperiam quasi iure error reprehenderit quisquam totam dolore aliquid debitis fugiat.",
                        "webUrl": "http://127.0.0.1:3000/groups/gitlab-org/-/epics/5"
                    },
                    "reference": "#3",
                    "state": "OPEN",
                    "title": "Consectetur nisi unde dolore cupiditate laborum et.",
                    "updated": "2025-08-04T09:32:02Z",
                    "updatedAt": "2025-08-04T09:32:02Z",
                    "webUrl": "http://127.0.0.1:3000/groups/gitlab-org/-/epics/3",
                    "widgets": null
                }
            ],
            "pageInfo": {
                "__typename": "PageInfo",
                "endCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wMy0wOCAxMjo0NToyMi4xNTgxMjcwMDAgKzAwMDAiLCJpZCI6IjYxOCJ9",
                "hasNextPage": false,
                "hasPreviousPage": false,
                "startCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wMy0wOCAxMjo0NToyMi4xNTgxMjcwMDAgKzAwMDAiLCJpZCI6IjYxOCJ9"
            }
        })
    );

    assert_eq!(
        output["fields"],
        json!([
            { "key": "title", "label": "Title", "name": "title" },
            { "key": "updated", "label": "Updated", "name": "updatedAt" },
            { "key": "epic", "label": "Epic", "name": "parent" }
        ])
    );
}

#[test]
fn test_transform_issues() {
    FeatureFlag::GlqlWorkItems.set(true);

    let data = serde_json::to_string(&json!({
        "group": {
            "issues": {
                "nodes": [
                    {
                        "id": "gid://gitlab/Issue/650",
                        "iid": "42",
                        "title": "epics test",
                        "webUrl": "http://127.0.0.1:3000/gitlab-org/gitlab-shell/-/issues/42",
                        "reference": "#42",
                        "state": "opened",
                        "type": "ISSUE",
                        "updatedAt": "2025-08-04T09:32:02Z",
                        "epic": {
                            "id": "gid://gitlab/Epic/8",
                            "iid": "3",
                            "reference": "&3",
                            "state": "opened",
                            "title": "Consectetur nisi unde dolore cupiditate laborum et.",
                            "webUrl": "http://127.0.0.1:3000/groups/gitlab-org/-/epics/3",
                            "__typename": "Epic"
                        },
                        "__typename": "Issue"
                    }
                ],
                "pageInfo": {
                    "startCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wNy0yNSAxODoxMzoyOC45NDc1NTAwMDAgKzAwMDAiLCJpZCI6IjY1MCJ9",
                    "endCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wNy0yNSAxODoxMzoyOC45NDc1NTAwMDAgKzAwMDAiLCJpZCI6IjY1MCJ9",
                    "hasNextPage": true,
                    "hasPreviousPage": false,
                    "__typename": "PageInfo"
                },
                "count": 4,
                "__typename": "IssueConnection"
            },
            "__typename": "Group"
        }
    })).unwrap();

    let output = transform_data(&data, r#"type, title, updated, epic"#);

    assert_eq!(
        output["data"],
        json!({
            "__typename": "IssueConnection",
            "count": 4,
            "nodes": [
                {
                    "__typename": "Issue",
                    "epic": {
                        "__typename": "Epic",
                        "id": "gid://gitlab/Epic/8",
                        "iid": "3",
                        "reference": "&3",
                        "state": "opened",
                        "title": "Consectetur nisi unde dolore cupiditate laborum et.",
                        "webUrl": "http://127.0.0.1:3000/groups/gitlab-org/-/epics/3"
                    },
                    "id": "gid://gitlab/Issue/650",
                    "iid": "42",
                    "reference": "#42",
                    "state": "opened",
                    "title": "epics test",
                    "type": "ISSUE",
                    "updated": "2025-08-04T09:32:02Z",
                    "updatedAt": "2025-08-04T09:32:02Z",
                    "webUrl": "http://127.0.0.1:3000/gitlab-org/gitlab-shell/-/issues/42"
                }
            ],
            "pageInfo": {
                "__typename": "PageInfo",
                "endCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wNy0yNSAxODoxMzoyOC45NDc1NTAwMDAgKzAwMDAiLCJpZCI6IjY1MCJ9",
                "hasNextPage": true,
                "hasPreviousPage": false,
                "startCursor": "eyJjcmVhdGVkX2F0IjoiMjAyNS0wNy0yNSAxODoxMzoyOC45NDc1NTAwMDAgKzAwMDAiLCJpZCI6IjY1MCJ9"
            }
        })
    );

    assert_eq!(
        output["fields"],
        json!([
            { "key": "type", "label": "Type", "name": "type" },
            { "key": "title", "label": "Title", "name": "title" },
            { "key": "updated", "label": "Updated", "name": "updatedAt" },
            { "key": "epic", "label": "Epic", "name": "epic" }
        ])
    );
}
