pub mod test_helpers;

use crate::test_helpers::*;
use serde_json::json;

fn mock_data_merge_requests() -> String {
    serde_json::to_string(&json!({
        "project": {
            "from_2024_08_09_to_2024_09_01": {
                "count": 1876,
                "totalTimeToMerge": 1194422204.332011
            },
            "from_2024_09_01_to_2024_10_01": {
                "count": 2650,
                "totalTimeToMerge": 1761566797.487367
            },
            "from_2024_10_01_to_2024_11_01": {
                "count": 3103,
                "totalTimeToMerge": 2086576128.855193
            },
            "from_2024_11_01_to_2024_12_01": {
                "count": 2542,
                "totalTimeToMerge": 1662207119.005869
            },
            "from_2024_12_01_to_2025_01_01": {
                "count": 2037,
                "totalTimeToMerge": 1380841068.758804
            },
            "from_2025_01_01_to_2025_02_01": {
                "count": 2572,
                "totalTimeToMerge": 2251245834.710288
            },
            "from_2025_02_01_to_2025_03_01": {
                "count": 2716,
                "totalTimeToMerge": 1946223159.426047
            },
            "from_2025_03_01_to_2025_04_01": {
                "count": 2894,
                "totalTimeToMerge": 2020946011.821066
            },
            "from_2025_04_01_to_2025_05_01": {
                "count": 2993,
                "totalTimeToMerge": 2021312509.793126
            },
            "from_2025_05_01_to_2025_06_01": {
                "count": 2649,
                "totalTimeToMerge": 1834820303.043201
            },
            "from_2025_06_01_to_2025_07_01": {
                "count": 2762,
                "totalTimeToMerge": 2034516437.573138
            },
            "from_2025_07_01_to_2025_08_01": {
                "count": 3114,
                "totalTimeToMerge": 1926510508.380342
            },
            "from_2025_08_01_to_2025_08_08": {
                "count": 667,
                "totalTimeToMerge": 451269078.168328
            }
        }
    }))
    .unwrap()
}

fn mock_data_issues() -> String {
    serde_json::to_string(&json!({
        "project": {
            "__typename": "Project",
            "from_2024_08_08_to_2024_09_01": {
                "__typename": "IssueConnection",
                "count": 0
            },
            "from_2024_09_01_to_2024_10_01": {
                "__typename": "IssueConnection",
                "count": 0
            },
            "from_2024_10_01_to_2024_11_01": {
                "__typename": "IssueConnection",
                "count": 0
            },
            "from_2024_11_01_to_2024_12_01": {
                "__typename": "IssueConnection",
                "count": 0
            },
            "from_2024_12_01_to_2025_01_01": {
                "__typename": "IssueConnection",
                "count": 0
            },
            "from_2025_01_01_to_2025_02_01": {
                "__typename": "IssueConnection",
                "count": 0
            },
            "from_2025_02_01_to_2025_03_01": {
                "__typename": "IssueConnection",
                "count": 2
            },
            "from_2025_03_01_to_2025_04_01": {
                "__typename": "IssueConnection",
                "count": 35
            },
            "from_2025_04_01_to_2025_05_01": {
                "__typename": "IssueConnection",
                "count": 0
            },
            "from_2025_05_01_to_2025_06_01": {
                "__typename": "IssueConnection",
                "count": 0
            },
            "from_2025_06_01_to_2025_07_01": {
                "__typename": "IssueConnection",
                "count": 0
            },
            "from_2025_07_01_to_2025_08_01": {
                "__typename": "IssueConnection",
                "count": 5
            },
            "from_2025_08_01_to_2025_08_08": {
                "__typename": "IssueConnection",
                "count": 2
            }
        }
    }))
    .unwrap()
}

#[test]
fn test_transform_aggregated_data_merge_requests() {
    let data = mock_data_merge_requests();
    let result = transform_aggregated_data(
        &data,
        "timeSegment(1m) on mergedAt as 'Date merged'",
        "count as 'Total count', totalTimeToMerge as 'Total time to merge'",
    );
    let expected_data = json!({
        "__typename": "GlqlAggregatedDataConnection",
        "count": 13,
        "nodes": [
            {
                "__typename": "GlqlAggregatedData",
                "count": 1876,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2024-08-09 to 2024-09-01"
                },
                "totalTimeToMerge": 1194422204.332011
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 2650,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2024-09-01 to 2024-10-01"
                },
                "totalTimeToMerge": 1761566797.487367
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 3103,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2024-10-01 to 2024-11-01"
                },
                "totalTimeToMerge": 2086576128.855193
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 2542,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2024-11-01 to 2024-12-01"
                },
                "totalTimeToMerge": 1662207119.005869
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 2037,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2024-12-01 to 2025-01-01"
                },
                "totalTimeToMerge": 1380841068.758804
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 2572,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2025-01-01 to 2025-02-01"
                },
                "totalTimeToMerge": 2251245834.710288
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 2716,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2025-02-01 to 2025-03-01"
                },
                "totalTimeToMerge": 1946223159.426047
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 2894,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2025-03-01 to 2025-04-01"
                },
                "totalTimeToMerge": 2020946011.821066
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 2993,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2025-04-01 to 2025-05-01"
                },
                "totalTimeToMerge": 2021312509.793126
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 2649,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2025-05-01 to 2025-06-01"
                },
                "totalTimeToMerge": 1834820303.043201
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 2762,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2025-06-01 to 2025-07-01"
                },
                "totalTimeToMerge": 2034516437.573138
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 3114,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2025-07-01 to 2025-08-01"
                },
                "totalTimeToMerge": 1926510508.380342
            },
            {
                "__typename": "GlqlAggregatedData",
                "count": 667,
                "mergedAt": {
                    "__typename": "GlqlDimension",
                    "title": "From 2025-08-01 to 2025-08-08"
                },
                "totalTimeToMerge": 451269078.168328
            }
        ]
    });
    let expected_fields = json!([
        {
            "key": "mergedAt",
            "label": "Date merged",
            "name": "mergedAt"
        },
        {
            "key": "count",
            "label": "Total count",
            "name": "count"
        },
        {
            "key": "totalTimeToMerge",
            "label": "Total time to merge",
            "name": "totalTimeToMerge"
        }
    ]);

    assert_eq!(result["data"], expected_data);
    assert_eq!(result["fields"], expected_fields);
}

#[test]
fn test_transform_aggregated_data_issues() {
    let data = mock_data_issues();
    let result = transform_aggregated_data(
        &data,
        "timeSegment(1m) on created as 'Date created'",
        "count",
    );

    assert_eq!(
        result["data"],
        json!({
            "__typename": "GlqlAggregatedDataConnection",
            "count": 13,
            "nodes": [
                {
                    "__typename": "IssueConnection",
                    "count": 0,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2024-08-08 to 2024-09-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2024-08-08 to 2024-09-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 0,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2024-09-01 to 2024-10-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2024-09-01 to 2024-10-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 0,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2024-10-01 to 2024-11-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2024-10-01 to 2024-11-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 0,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2024-11-01 to 2024-12-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2024-11-01 to 2024-12-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 0,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2024-12-01 to 2025-01-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2024-12-01 to 2025-01-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 0,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-01-01 to 2025-02-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-01-01 to 2025-02-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 2,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-02-01 to 2025-03-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-02-01 to 2025-03-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 35,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-03-01 to 2025-04-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-03-01 to 2025-04-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 0,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-04-01 to 2025-05-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-04-01 to 2025-05-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 0,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-05-01 to 2025-06-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-05-01 to 2025-06-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 0,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-06-01 to 2025-07-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-06-01 to 2025-07-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 5,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-07-01 to 2025-08-01"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-07-01 to 2025-08-01"
                    }
                },
                {
                    "__typename": "IssueConnection",
                    "count": 2,
                    "created": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-08-01 to 2025-08-08"
                    },
                    "createdAt": {
                        "__typename": "GlqlDimension",
                        "title": "From 2025-08-01 to 2025-08-08"
                    }
                }
            ]
        })
    );
    assert_eq!(
        result["fields"],
        json!([
            {
                "key": "created",
                "label": "Date created",
                "name": "createdAt"
            },
            {
                "key": "count",
                "label": "Count",
                "name": "count"
            }
        ])
    );
}
