use chrono::{DateTime, NaiveDateTime, Utc};
use glql::types::{Context, Source};
use glql::{compile, compile_glql, transform};
use serde_json::{Value as JsonValue, json};

pub fn compile_graphql(query: &str) -> String {
    let context = json!({
        "fields": "id, title",
    });

    let result = serde_json::from_str::<JsonValue>(&compile(query, &context.to_string())).unwrap();
    result["output"].as_str().unwrap().to_string()
}

pub fn compile_with_fields(query: &str, fields: &[&str]) -> String {
    let context = json!({
        "fields": fields.join(","),
    });

    let result = serde_json::from_str::<JsonValue>(&compile(query, &context.to_string())).unwrap();
    result["output"].as_str().unwrap().to_string()
}

pub fn compile_with_aggregate(query: &str, dimensions: &str, metrics: &str) -> String {
    let context = json!({
        "aggregate": { "dimensions": dimensions, "metrics": metrics },
        "fields": "id, title",
    });

    let result = serde_json::from_str::<JsonValue>(&compile(query, &context.to_string())).unwrap();
    result["output"].as_str().unwrap().to_string()
}

pub fn compile_with_time_from_string(reference_time: &str, query: &str) -> String {
    let dt =
        NaiveDateTime::parse_from_str(&format!("{reference_time} 00:00:00"), "%Y-%m-%d %H:%M:%S")
            .unwrap();
    let mut context = Context {
        time: DateTime::from_naive_utc_and_offset(dt, Utc),
        ..Context::default()
    };

    compile_glql(query, &mut context).output
}

pub fn compile_with_user_context(username: &str, query: &str) -> String {
    let context = json!({
        "username": username,
        "fields": "id, title",
    });

    let result = serde_json::from_str::<JsonValue>(&compile(query, &context.to_string())).unwrap();
    result["output"].as_str().unwrap().to_string()
}

pub fn compile_with_group_context(group: &str, query: &str) -> String {
    let context = json!({
        "group": group,
        "fields": "id, title",
    });

    let result = serde_json::from_str::<JsonValue>(&compile(query, &context.to_string())).unwrap();
    result["output"].as_str().unwrap().to_string()
}

pub fn has_fields_in_response(field_name: &str, fields: &[&str], query: &str) -> bool {
    let response = compile_graphql(query);
    let expected = format!("{} {{\n{}\n}}", field_name, fields.join("\n"));
    response.contains(&expected)
}

pub fn has_fields_in_nodes_response(field_name: &str, fields: &[&str], query: &str) -> bool {
    let response = compile_graphql(query);
    let expected = format!("{} {{\nnodes {{\n{}\n}}\n}}", field_name, fields.join("\n"));
    response.contains(&expected)
}

pub fn compile_with_project_context(project: &str, query: &str) -> String {
    let context = json!({
        "project": project,
        "fields": "id, title",
    });

    let result = serde_json::from_str::<JsonValue>(&compile(query, &context.to_string())).unwrap();
    result["output"].as_str().unwrap().to_string()
}

pub fn compile_with_source(source: Source, query: &str) -> String {
    let type_name = match source {
        Source::Issues => "issue",
        Source::MergeRequests => "mergerequest",
        Source::Epics => "epic",
    };

    let context = json!({
        "fields": "id, title",
    });

    let query = format!("{query} and type = {type_name}");

    let result = serde_json::from_str::<JsonValue>(&compile(&query, &context.to_string())).unwrap();
    result["output"].as_str().unwrap().to_string()
}

pub fn compile_with_sort(query: &str, sort: &str) -> String {
    let context = json!({
        "sort": sort,
        "fields": "id, title",
    });

    let result = serde_json::from_str::<JsonValue>(&compile(query, &context.to_string())).unwrap();
    result["output"].as_str().unwrap().to_string()
}

pub fn transform_data(data: &str, fields: &str) -> JsonValue {
    let context = json!({ "fields": fields });
    let transformed = transform(data, &context.to_string());

    serde_json::from_str(&transformed).unwrap()
}

pub fn transform_aggregated_data(data: &str, dimensions: &str, metrics: &str) -> JsonValue {
    let context = json!({
        "aggregate": { "dimensions": dimensions, "metrics": metrics },
        "fields": "id, title",
    });
    let transformed = transform(data, &context.to_string());

    serde_json::from_str(&transformed).unwrap()
}
