pub mod test_helpers;

use crate::test_helpers::*;
use glql::utils::feature_flags::FeatureFlag;

macro_rules! test_success {
    ($($name:ident: $value:expr,)*) => { $(
        #[test]
        fn $name() {
            // test default
            let compiled = compile_with_sort(
                format!("group = \"gitlab-org\" and {}", $value.0).as_str(),
                $value.1,
            );
            assert_eq!(compiled.lines().nth(2).unwrap(), $value.2);

            // test ascending
            let compiled = compile_with_sort(
                format!("group = \"gitlab-org\" and {}", $value.0).as_str(),
                format!("{} asc", $value.1).as_str(),
            );
            assert_eq!(compiled.lines().nth(2).unwrap(), $value.2);

            // test descending
            let compiled = compile_with_sort(
                format!("group = \"gitlab-org\" and {}", $value.0).as_str(),
                format!("{} desc", $value.1).as_str(),
            );
            assert_eq!(compiled.lines().nth(2).unwrap(), $value.3);
        }
    )* }
}

macro_rules! test_success_work_items {
    ($($name:ident: $value:expr,)*) => { $(
        #[test]
        fn $name() {
            FeatureFlag::set(FeatureFlag::GlqlWorkItems, true);

            // test default
            let compiled = compile_with_sort(
                format!("group = \"gitlab-org\" and {}", $value.0).as_str(),
                $value.1,
            );
            assert_eq!(compiled.lines().nth(2).unwrap(), $value.2);

            // test ascending
            let compiled = compile_with_sort(
                format!("group = \"gitlab-org\" and {}", $value.0).as_str(),
                format!("{} asc", $value.1).as_str(),
            );
            assert_eq!(compiled.lines().nth(2).unwrap(), $value.2);

            // test descending
            let compiled = compile_with_sort(
                format!("group = \"gitlab-org\" and {}", $value.0).as_str(),
                format!("{} desc", $value.1).as_str(),
            );
            assert_eq!(compiled.lines().nth(2).unwrap(), $value.3);
        }
    )* }
}

macro_rules! test_error {
    ($($name:ident: $value:expr,)*) => { $(
        #[test]
        fn $name() {
            let compiled = compile_with_sort(format!("group = \"gitlab-org\" and {}", $value.0).as_str(), $value.1);

            assert_eq!(compiled.as_str(), $value.2);
        }
    )* }
}

test_success! {
    // MergeRequests
    test_merge_requests_sort_by_merged: (
        "type = MergeRequest and author = @foo",
        "merged",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: MERGED_AT_ASC, includeSubgroups: true) {",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: MERGED_AT_DESC, includeSubgroups: true) {",
    ),
    test_merge_requests_sort_by_closed: (
        "type = MergeRequest and author = @foo",
        "closed",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CLOSED_AT_ASC, includeSubgroups: true) {",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CLOSED_AT_DESC, includeSubgroups: true) {",
    ),
    test_merge_requests_sort_by_title: (
        "type = MergeRequest and author = @foo",
        "title",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: TITLE_ASC, includeSubgroups: true) {",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: TITLE_DESC, includeSubgroups: true) {",
    ),
    test_merge_requests_sort_by_popularity: (
        "type = MergeRequest and author = @foo",
        "popularity",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: POPULARITY_ASC, includeSubgroups: true) {",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: POPULARITY_DESC, includeSubgroups: true) {",
    ),
    test_merge_requests_sort_by_milestone: (
        "type = MergeRequest and author = @foo",
        "milestone",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: MILESTONE_DUE_ASC, includeSubgroups: true) {",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: MILESTONE_DUE_DESC, includeSubgroups: true) {",
    ),
    test_merge_requests_sort_by_updated: (
        "type = MergeRequest and author = @foo",
        "updated",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: UPDATED_ASC, includeSubgroups: true) {",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: UPDATED_DESC, includeSubgroups: true) {",
    ),
    test_merge_requests_sort_by_created: (
        "type = MergeRequest and author = @foo",
        "created",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CREATED_ASC, includeSubgroups: true) {",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CREATED_DESC, includeSubgroups: true) {",
    ),

    // Epics
    test_epics_sort_by_start: (
        "type = Epic and author = @foo",
        "start",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: START_DATE_ASC, includeDescendants: true, excludeProjects: true) {",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: START_DATE_DESC, includeDescendants: true, excludeProjects: true) {",
    ),
    test_epics_sort_by_due: (
        "type = Epic and author = @foo",
        "due",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: DUE_DATE_ASC, includeDescendants: true, excludeProjects: true) {",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: DUE_DATE_DESC, includeDescendants: true, excludeProjects: true) {",
    ),
    test_epics_sort_by_title: (
        "type = Epic and author = @foo",
        "title",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: TITLE_ASC, includeDescendants: true, excludeProjects: true) {",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: TITLE_DESC, includeDescendants: true, excludeProjects: true) {",
    ),
    test_epics_sort_by_created: (
        "type = Epic and author = @foo",
        "created",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CREATED_ASC, includeDescendants: true, excludeProjects: true) {",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CREATED_DESC, includeDescendants: true, excludeProjects: true) {",
    ),
    test_epics_sort_by_updated: (
        "type = Epic and author = @foo",
        "updated",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: UPDATED_ASC, includeDescendants: true, excludeProjects: true) {",
        "    workItems(types: EPIC, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: UPDATED_DESC, includeDescendants: true, excludeProjects: true) {",
    ),

    // Issues
    test_issues_sort_by_due_date: (
        "type = Issue and author = @foo",
        "due",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: DUE_DATE_ASC, includeSubgroups: true) {",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: DUE_DATE_DESC, includeSubgroups: true) {",
    ),
    test_issues_sort_by_title: (
        "type = Issue and author = @foo",
        "title",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: TITLE_ASC, includeSubgroups: true) {",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: TITLE_DESC, includeSubgroups: true) {",
    ),
    test_issues_sort_by_popularity: (
        "type = Issue and author = @foo",
        "popularity",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: POPULARITY_ASC, includeSubgroups: true) {",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: POPULARITY_DESC, includeSubgroups: true) {",
    ),
    test_issues_sort_by_closed: (
        "type = Issue and author = @foo",
        "closed",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CLOSED_AT_ASC, includeSubgroups: true) {",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CLOSED_AT_DESC, includeSubgroups: true) {",
    ),
    test_issues_sort_by_weight: (
        "type = Issue and author = @foo",
        "weight",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: WEIGHT_ASC, includeSubgroups: true) {",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: WEIGHT_DESC, includeSubgroups: true) {",
    ),
    test_issues_sort_by_health: (
        "type = Issue and author = @foo",
        "health",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: HEALTH_STATUS_ASC, includeSubgroups: true) {",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: HEALTH_STATUS_DESC, includeSubgroups: true) {",
    ),
    test_issues_sort_by_milestone: (
        "type = Issue and author = @foo",
        "milestone",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: MILESTONE_DUE_ASC, includeSubgroups: true) {",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: MILESTONE_DUE_DESC, includeSubgroups: true) {",
    ),
    test_issues_sort_by_updated: (
        "type = Issue and author = @foo",
        "updated",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: UPDATED_ASC, includeSubgroups: true) {",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: UPDATED_DESC, includeSubgroups: true) {",
    ),
    test_issues_sort_by_created: (
        "type = Issue and author = @foo",
        "created",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CREATED_ASC, includeSubgroups: true) {",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CREATED_DESC, includeSubgroups: true) {",
    ),

    // Aliases - we only test a few of them
    test_issues_sort_by_due_date_alias: (
        "type = Issue and author = @foo",
        "dueDate",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: DUE_DATE_ASC, includeSubgroups: true) {",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: DUE_DATE_DESC, includeSubgroups: true) {",
    ),

    test_issues_sort_by_created_alias: (
        "type = Issue and author = @foo",
        "createdAt",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CREATED_ASC, includeSubgroups: true) {",
        "    issues(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CREATED_DESC, includeSubgroups: true) {",
    ),

    test_merge_requests_sort_by_merged_alias: (
        "type = MergeRequest and author = @foo",
        "mergedAt",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: MERGED_AT_ASC, includeSubgroups: true) {",
        "    mergeRequests(authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: MERGED_AT_DESC, includeSubgroups: true) {",
    ),
}

test_success_work_items! {
    test_issues_sort_by_start_date_work_items: (
        "type = Issue and author = @foo",
        "start",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: START_DATE_ASC, includeDescendants: true) {",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: START_DATE_DESC, includeDescendants: true) {",
    ),
    test_issues_sort_by_due_date_work_items: (
        "type = Issue and author = @foo",
        "due",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: DUE_DATE_ASC, includeDescendants: true) {",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: DUE_DATE_DESC, includeDescendants: true) {",
    ),
    test_issues_sort_by_title_work_items: (
        "type = Issue and author = @foo",
        "title",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: TITLE_ASC, includeDescendants: true) {",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: TITLE_DESC, includeDescendants: true) {",
    ),
    test_issues_sort_by_popularity_work_items: (
        "type = Issue and author = @foo",
        "popularity",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: POPULARITY_ASC, includeDescendants: true) {",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: POPULARITY_DESC, includeDescendants: true) {",
    ),
    test_issues_sort_by_closed_work_items: (
        "type = Issue and author = @foo",
        "closed",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CLOSED_AT_ASC, includeDescendants: true) {",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CLOSED_AT_DESC, includeDescendants: true) {",
    ),
    test_issues_sort_by_weight_work_items: (
        "type = Issue and author = @foo",
        "weight",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: WEIGHT_ASC, includeDescendants: true) {",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: WEIGHT_DESC, includeDescendants: true) {",
    ),
    test_issues_sort_by_health_work_items: (
        "type = Issue and author = @foo",
        "health",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: HEALTH_STATUS_ASC, includeDescendants: true) {",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: HEALTH_STATUS_DESC, includeDescendants: true) {",
    ),
    test_issues_sort_by_milestone_work_items: (
        "type = Issue and author = @foo",
        "milestone",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: MILESTONE_DUE_ASC, includeDescendants: true) {",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: MILESTONE_DUE_DESC, includeDescendants: true) {",
    ),
    test_issues_sort_by_updated_work_items: (
        "type = Issue and author = @foo",
        "updated",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: UPDATED_ASC, includeDescendants: true) {",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: UPDATED_DESC, includeDescendants: true) {",
    ),
    test_issues_sort_by_created_work_items  : (
        "type = Issue and author = @foo",
        "created",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CREATED_ASC, includeDescendants: true) {",
        "    workItems(types: ISSUE, authorUsername: \"foo\", before: $before, after: $after, first: $limit, sort: CREATED_DESC, includeDescendants: true) {",
    ),
}

test_error! {
    test_merge_requests_sort_by_invalid: (
        "type = MergeRequest and author = @foo",
        "invalid",
        "Error: `invalid` is not a recognized sort field for merge requests. Supported fields to sort by: `merged`, `closed`, `title`, `popularity`, `milestone`, `updated`, `created`."
    ),

    test_epics_sort_by_invalid: (
        "type = Epic and author = @foo",
        "invalid",
        "Error: `invalid` is not a recognized sort field for work items. Supported fields to sort by: `due`, `title`, `popularity`, `closed`, `weight`, `health`, `milestone`, `updated`, `created`, `start`."
    ),

    test_issues_sort_by_invalid: (
        "type = Issue and author = @foo",
        "invalid",
        "Error: `invalid` is not a recognized sort field for work items. Supported fields to sort by: `due`, `title`, `popularity`, `closed`, `weight`, `health`, `milestone`, `updated`, `created`."
    ),
}
