pub mod test_helpers;

use crate::test_helpers::*;

#[test]
fn test_invalid_field() {
    assert_eq!(
        compile_graphql("foo = \"foo\""),
        "Error: foo is not a recognized field."
    );
}

#[test]
fn test_invalid_weight_operator() {
    assert_eq!(
        compile_graphql("weight > 1"),
        "Error: `weight` does not support the greater than (`>`) operator. Supported operators: equals (`=`), not equals (`!=`)."
    );
}

#[test]
fn test_invalid_in_with_non_array() {
    assert_eq!(
        compile_graphql("label IN \"fish\""),
        "Error: `label` does not support the is one of (`in`) operator for `\"fish\"`. Supported operators: equals (`=`), not equals (`!=`)."
    );
}

#[test]
fn test_invalid_in_with_empty_array() {
    assert_eq!(
        compile_graphql("label IN ()"),
        "Error: `label` does not support the is one of (`in`) operator for `()`. Supported operators: equals (`=`), not equals (`!=`)."
    );
}

#[test]
fn test_invalid_gt_with_array() {
    assert_eq!(
        compile_graphql("label > ()"),
        "Error: `label` does not support the greater than (`>`) operator. Supported operators: equals (`=`), not equals (`!=`), is one of (`in`)."
    );
}

#[test]
fn test_invalid_lt_with_array() {
    assert_eq!(
        compile_graphql("label < ()"),
        "Error: `label` does not support the less than (`<`) operator. Supported operators: equals (`=`), not equals (`!=`), is one of (`in`)."
    );
}

#[test]
fn test_invalid_group_with_not_equals() {
    assert_eq!(
        compile_graphql("group != \"fish\""),
        "Error: `group` does not support the not equals (`!=`) operator. Supported operators: equals (`=`)."
    );
}

#[test]
fn test_invalid_project_with_not_equals() {
    assert_eq!(
        compile_graphql("project != \"fish\""),
        "Error: `project` does not support the not equals (`!=`) operator. Supported operators: equals (`=`)."
    );
}

#[test]
fn test_invalid_created_with_not_equals() {
    assert_eq!(
        compile_graphql("created != \"fish\""),
        "Error: `created` does not support the not equals (`!=`) operator. Supported operators: equals (`=`), greater than (`>`), greater than or equal to (`>=`), less than (`<`), less than or equal to (`<=`)."
    );
}

#[test]
fn test_invalid_use_of_list_with_field() {
    assert_eq!(
        compile_graphql("created = (\"fish\")"),
        "Error: `created` cannot be compared with `(\"fish\")`. Supported value types: `Date`."
    );
}

#[test]
fn test_invalid_health_status() {
    assert_eq!(
        compile_graphql("health = \"foo\""),
        "Error: `health` cannot be compared with `\"foo\"`. Supported value types: `StringEnum` (`\"on track\"`, `\"needs attention\"`, `\"at risk\"`), `Nullable` (`null`, `none`, `any`)."
    );
}

#[test]
fn test_valid_use_of_list_with_milestone() {
    assert_eq!(
        compile_graphql("milestone in (\"fish\")")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(milestoneTitle: [\"fish\"], before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_milestone_not_equal_to_none_becomes_equal_to_any() {
    assert_eq!(
        compile_graphql("milestone != none").lines().nth(1).unwrap(),
        "  issues(milestoneWildcardId: ANY, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_milestone_not_equal_to_any_becomes_equal_to_none() {
    assert_eq!(
        compile_graphql("milestone != any").lines().nth(1).unwrap(),
        "  issues(milestoneWildcardId: NONE, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_confidential_only_boolean() {
    assert_eq!(
        compile_graphql("confidential = ()"),
        "Error: `confidential` cannot be compared with `()`. Supported value types: `Boolean` (`true`, `false`)."
    );
}

#[test]
fn test_invalid_function() {
    assert_eq!(
        compile_graphql("milestone = catch(\"fish\")"),
        "Error: Unrecognized function: catch(\"fish\")"
    );
}

#[test]
fn test_multiple_in_for_same_field() {
    assert_eq!(
        compile_graphql("label IN (\"foo\") AND label IN (\"bar\")"),
        "Error: The is one of (`in`) operator can only be used once with `label`."
    );
}

#[test]
fn test_field_name_check_precedence() {
    assert_eq!(
        compile_graphql("foo = (\"foo\")"),
        "Error: foo is not a recognized field."
    );
}

#[test]
fn test_null_is_falsey() {
    assert_eq!(
        compile_graphql("confidential = null")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(confidential: false, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_false_string_is_falsey() {
    assert_eq!(
        compile_graphql("confidential = \"false\"")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(confidential: false, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_false_string_is_falsey_case_insensitive() {
    assert_eq!(
        compile_graphql("confidential = \"faLse\"")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(confidential: false, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_zero_is_falsey() {
    assert_eq!(
        compile_graphql("confidential = 0").lines().nth(1).unwrap(),
        "  issues(confidential: false, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_true_string_is_truthy() {
    assert_eq!(
        compile_graphql("confidential = \"true\"")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(confidential: true, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_true_string_is_truthy_case_insensitive() {
    assert_eq!(
        compile_graphql("confidential = \"tRue\"")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(confidential: true, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_one_is_truthy() {
    assert_eq!(
        compile_graphql("confidential = 1").lines().nth(1).unwrap(),
        "  issues(confidential: true, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_created_with_equals_is_expanded() {
    assert_eq!(
        compile_graphql("created = 2024-01-01")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(createdAfter: \"2024-01-01 00:00\", createdBefore: \"2024-01-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_created_with_equals_is_expanded_with_string() {
    assert_eq!(
        compile_graphql("created = \"2024-01-01\"")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(createdAfter: \"2024-01-01 00:00\", createdBefore: \"2024-01-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_updated_with_equals_is_expanded() {
    assert_eq!(
        compile_graphql("updated = 2024-01-01")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(updatedAfter: \"2024-01-01 00:00\", updatedBefore: \"2024-01-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_updated_with_equals_is_expanded_with_string() {
    assert_eq!(
        compile_graphql("updated = \"2024-01-01\"")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(updatedAfter: \"2024-01-01 00:00\", updatedBefore: \"2024-01-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_opened_with_equals_is_expanded() {
    assert_eq!(
        compile_graphql("opened = 2024-01-01")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(createdAfter: \"2024-01-01 00:00\", createdBefore: \"2024-01-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_opened_with_equals_is_expanded_with_string() {
    assert_eq!(
        compile_graphql("opened = \"2024-01-01\"")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(createdAfter: \"2024-01-01 00:00\", createdBefore: \"2024-01-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_closed_with_equals_is_expanded() {
    assert_eq!(
        compile_graphql("closed = 2024-01-01")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(closedAfter: \"2024-01-01 00:00\", closedBefore: \"2024-01-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_closed_with_equals_is_expanded_with_string() {
    assert_eq!(
        compile_graphql("closed = \"2024-01-01\"")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(closedAfter: \"2024-01-01 00:00\", closedBefore: \"2024-01-01 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_timeable_with_equals_is_expanded_with_relative_time() {
    assert_eq!(
        compile_with_time_from_string("2024-07-12", "closed = -1d")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(closedAfter: \"2024-07-11 00:00\", closedBefore: \"2024-07-11 23:59\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_strict_timeable_with_equals_error_on_non_date() {
    assert_eq!(
        compile_graphql("created = 1"),
        "Error: `created` cannot be compared with `1`. Supported value types: `Date`."
    );
}

#[test]
fn test_timeable_with_equals_error_on_non_bool() {
    assert_eq!(
        compile_graphql("opened = \"fish\""),
        "Error: `opened` cannot be compared with `\"fish\"`. Supported value types: `Date`."
    );
}

#[test]
fn test_malformed_date_format_slash() {
    assert_eq!(
        compile_graphql("created = 2024/01/01"),
        "Error: Invalid date format '2024/01/01'. Expected format: yyyy-mm-dd"
    );
}

#[test]
fn test_malformed_date_format_dot() {
    assert_eq!(
        compile_graphql("created = 2024.01.01"),
        "Error: Invalid date format '2024.01.01'. Expected format: yyyy-mm-dd"
    );
}

#[test]
fn test_invalid_relative_date_unit_hours() {
    assert_eq!(
        compile_graphql("created > 1h"),
        "Error: Unexpected character `h`. Expected d (day), w (week), m (month), or y (year)"
    );
}

#[test]
fn test_invalid_relative_date_unit_seconds() {
    assert_eq!(
        compile_graphql("created > 1s"),
        "Error: Unexpected character `s`. Expected d (day), w (week), m (month), or y (year)"
    );
}

#[test]
fn test_malformed_username_reference_double_at() {
    assert_eq!(
        compile_graphql("assignee = @user@name"),
        "Error: Invalid username reference `@user@name`"
    );
}

#[test]
fn test_malformed_username_reference_numeric() {
    assert_eq!(
        compile_graphql("assignee = @123"),
        "Error: Invalid username reference `@123`"
    );
}

#[test]
fn test_nested_parentheses() {
    assert_eq!(
        compile_graphql("label = ((\"foo\"))"),
        "Error: Unexpected token near `(\"foo\"))`"
    );
}

#[test]
fn test_escaped_quotes_in_strings() {
    assert_eq!(
        compile_graphql("label = \"foo\\\"bar\"")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(labelName: \"foo\\\"bar\", before: $before, after: $after, first: $limit) {"
    );
}
