pub mod test_helpers;

use crate::test_helpers::*;

#[test]
fn test_empty_filter() {
    assert_eq!(
        compile_graphql("weight=1").lines().nth(1).unwrap(),
        "  issues(weight: \"1\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_project_scope() {
    let result = compile_graphql("project=\"gitlab-org/gitlab\"");
    let lines: Vec<&str> = result.lines().collect();
    assert_eq!(lines[1], "  project(fullPath: \"gitlab-org/gitlab\") {");
    assert_eq!(
        lines[2],
        "    issues(before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_group_scope() {
    let result = compile_graphql("group=\"gitlab-org\"");
    let lines: Vec<&str> = result.lines().collect();
    assert_eq!(lines[1], "  group(fullPath: \"gitlab-org\") {");
    assert_eq!(
        lines[2],
        "    issues(before: $before, after: $after, first: $limit, includeSubgroups: true) {"
    );
}

#[test]
fn test_include_subgroups() {
    assert_eq!(
        compile_graphql("group = \"gitlab-org\" and includeSubgroups = true")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(includeSubgroups: true, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_include_subgroups_false() {
    assert_eq!(
        compile_graphql("group = \"gitlab-org\" and includeSubgroups = false")
            .lines()
            .nth(2)
            .unwrap(),
        "    issues(includeSubgroups: false, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_invalid_include_subgroups_with_project() {
    assert_eq!(
        compile_graphql("project = \"gitlab-org/gitlab\" and includeSubgroups = true"),
        "Error: `includeSubgroups` can only be used with: `group`."
    );
}

#[test]
fn test_include_subgroup_invalid_value() {
    assert_eq!(
        compile_graphql("group=\"gitlab-org\" and includeSubgroups = \"wrong\""),
        "Error: `includeSubgroups` cannot be compared with `\"wrong\"`. Supported value types: `Boolean` (`true`, `false`)."
    )
}

#[test]
fn test_group_and_project_scope() {
    assert_eq!(
        compile_graphql("group=\"gitlab-org\" and project=\"gitlab-org/gitlab\"")
            .lines()
            .nth(1)
            .unwrap(),
        "  project(fullPath: \"gitlab-org/gitlab\") {"
    );
}
