pub mod test_helpers;

use crate::test_helpers::*;

#[test]
fn test_missing_field() {
    assert_eq!(
        compile_graphql("").lines().next().unwrap(),
        "Error: Unknown error. Is there a string to parse?"
    );
}

#[test]
fn test_malformed_field() {
    assert_eq!(
        compile_graphql("confid3ntial = true")
            .lines()
            .next()
            .unwrap(),
        "Error: Unexpected `3ntial = true`, expected operator (one of IN, =, !=, >, or <)"
    );
}

#[test]
fn test_missing_operator() {
    assert_eq!(
        compile_graphql("created").lines().next().unwrap(),
        "Error: Unexpected end of input, expected operator (one of IN, =, !=, >, or <)"
    );
}

#[test]
fn test_unrecognized_operator() {
    assert_eq!(
        compile_graphql("created * today()").lines().next().unwrap(),
        "Error: Unexpected `* today()`, expected operator (one of IN, =, !=, >, or <)"
    );
}

#[test]
fn test_missing_value() {
    assert_eq!(
        compile_graphql("created >").lines().next().unwrap(),
        "Error: Unexpected end of input, expected valid value"
    );
}

#[test]
fn test_unrecognized_value() {
    assert_eq!(
        compile_graphql("created > foo and weight = 1")
            .lines()
            .next()
            .unwrap(),
        "Error: Unexpected term `foo`"
    );
}

#[test]
fn test_double_quoted_strings() {
    assert_eq!(
        compile_graphql("milestone = \"17.5\"")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(milestoneTitle: \"17.5\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_single_quoted_strings() {
    assert_eq!(
        compile_graphql("milestone = '17.5'")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(milestoneTitle: \"17.5\", before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_mismatch_quoted_strings() {
    assert_eq!(
        compile_graphql("milestone = \"17.5'"),
        "Error: Unterminated string `\"17.5'`"
    );

    assert_eq!(
        compile_graphql("milestone = \"with escaped quotes\\\""),
        "Error: Unterminated string `\"with escaped quotes\\\"`"
    );

    assert_eq!(
        compile_graphql("milestone = '17.5\""),
        "Error: Unterminated string `'17.5\"`"
    );

    assert_eq!(
        compile_graphql("milestone = 'with escaped quotes\\'"),
        "Error: Unterminated string `'with escaped quotes\\'`"
    );
}

#[test]
fn test_value_with_additional_characters() {
    assert_eq!(
        compile_graphql("confidential = true1bar and weight = 1")
            .lines()
            .next()
            .unwrap(),
        "Error: Unexpected token at end of input: `1bar and weight = 1`"
    );
}

#[test]
fn test_null_with_additional_characters() {
    assert_eq!(
        compile_graphql("author = null1 and weight = 1")
            .lines()
            .next()
            .unwrap(),
        "Error: Unexpected token at end of input: `1 and weight = 1`"
    );
}

#[test]
fn test_array_with_tokens() {
    assert_eq!(
        compile_graphql("label in (~true, ~false)")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(or: {labelNames: [\"true\", \"false\"]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_array_with_spaces() {
    assert_eq!(
        compile_graphql("label in (   ~true    ,    ~false    )")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(or: {labelNames: [\"true\", \"false\"]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_leading_whitespace() {
    assert_eq!(
        compile_graphql("  \n    label in (~true, ~false)")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(or: {labelNames: [\"true\", \"false\"]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_trailing_whitespace() {
    assert_eq!(
        compile_graphql("label in (~true, ~false)  \n    ")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(or: {labelNames: [\"true\", \"false\"]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_value_with_additional_characters_at_end() {
    assert_eq!(
        compile_graphql("confidential = true1")
            .lines()
            .next()
            .unwrap(),
        "Error: Unexpected token at end of input: `1`"
    );
}

#[test]
fn test_null_with_additional_characters_at_end() {
    assert_eq!(
        compile_graphql("author = null1").lines().next().unwrap(),
        "Error: Unexpected token at end of input: `1`"
    );
}

#[test]
fn test_value_with_additional_characters_in_array() {
    assert_eq!(
        compile_graphql("label in (true1, false)")
            .lines()
            .next()
            .unwrap(),
        "Error: Unexpected token near `1, false)`"
    );
}

#[test]
fn test_value_with_additional_characters_at_end_of_array() {
    assert_eq!(
        compile_graphql("label in (true, false1)")
            .lines()
            .next()
            .unwrap(),
        "Error: Unexpected token near `1)`"
    );
}

#[test]
fn test_unexpected_token_at_end_of_input() {
    assert_eq!(
        compile_graphql("author=currentUser() foo")
            .lines()
            .next()
            .unwrap(),
        "Error: Unexpected token at end of input: `foo`"
    );
}

#[test]
fn test_value_with_additional_characters_in_array_with_spaces() {
    assert_eq!(
        compile_graphql("label in (   true1 ,  false   )")
            .lines()
            .next()
            .unwrap(),
        "Error: Unexpected token near `1 ,  false   )`"
    );
}

#[test]
fn test_value_with_additional_characters_at_end_of_array_with_spaces() {
    assert_eq!(
        compile_graphql("label in (   true,   false1 )")
            .lines()
            .next()
            .unwrap(),
        "Error: Unexpected token near `1 )`"
    );
}

#[test]
fn test_array_with_tokens_with_bracket_spaces() {
    assert_eq!(
        compile_graphql("label in ( ~true, ~false )")
            .lines()
            .nth(1)
            .unwrap(),
        "  issues(or: {labelNames: [\"true\", \"false\"]}, before: $before, after: $after, first: $limit) {"
    );
}

#[test]
fn test_invalid_date_literal() {
    assert_eq!(
        compile_graphql("created > 7x").lines().next().unwrap(),
        "Error: Unexpected character `x`. Expected d (day), w (week), m (month), or y (year)"
    );
}
