pub mod test_helpers;
use crate::test_helpers::compile_graphql;
use glql::compile;
use glql::utils::feature_flags::FeatureFlag;
use serde_json::{Value as JsonValue, json};

macro_rules! test {
    ($($name:ident: ($input:expr, $expected_issues:expr, $expected_epics:expr, $expected_work_items:expr),)*) => { $(
        #[test]
        fn $name() {
            // test issues
            let compiled = compile_graphql(concat!("project = \"gitlab-org/gitlab\" and ", $input));
            assert_eq!(compiled.lines().nth(2).unwrap_or_else(|| compiled.as_str()), $expected_issues);

            // test epics
            let compiled = compile_graphql(concat!("group = \"gitlab-org\" and type = Epic and ", $input));
            assert_eq!(compiled.lines().nth(2).unwrap_or_else(|| compiled.as_str()), $expected_epics);

            // test work items
            FeatureFlag::GlqlWorkItems.set(true);
            let compiled = compile_graphql(concat!("project = \"gitlab-org/gitlab\" and ", $input));
            assert_eq!(compiled.lines().nth(2).unwrap_or_else(|| compiled.as_str()), $expected_work_items);
        }
    )* }
}

test! {
    test_subscribed: (
        "subscribed = true",
        "    issues(subscribed: EXPLICITLY_SUBSCRIBED, before: $before, after: $after, first: $limit) {",
        "    workItems(types: EPIC, subscribed: EXPLICITLY_SUBSCRIBED, before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {",
        "    workItems(subscribed: EXPLICITLY_SUBSCRIBED, before: $before, after: $after, first: $limit) {"
    ),

    test_unsubscribed: (
        "subscribed = false",
        "    issues(subscribed: EXPLICITLY_UNSUBSCRIBED, before: $before, after: $after, first: $limit) {",
        "    workItems(types: EPIC, subscribed: EXPLICITLY_UNSUBSCRIBED, before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {",
        "    workItems(subscribed: EXPLICITLY_UNSUBSCRIBED, before: $before, after: $after, first: $limit) {"
    ),

    test_subscribed_invalid: (
        "subscribed = \"invalid\"",
        "Error: `subscribed` cannot be compared with `\"invalid\"`. Supported value types: `Boolean` (`true`, `false`).",
        "Error: `subscribed` cannot be compared with `\"invalid\"`. Supported value types: `Boolean` (`true`, `false`).",
        "Error: `subscribed` cannot be compared with `\"invalid\"`. Supported value types: `Boolean` (`true`, `false`)."
    ),

    test_my_reaction_equals: (
        "myreaction = \"thumbsup\"",
        "    issues(myReactionEmoji: \"thumbsup\", before: $before, after: $after, first: $limit) {",
        "    workItems(types: EPIC, myReactionEmoji: \"thumbsup\", before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {",
        "    workItems(myReactionEmoji: \"thumbsup\", before: $before, after: $after, first: $limit) {"
    ),

    test_my_reaction_not_equals: (
        "myreaction != \"thumbsup\"",
        "    issues(not: {myReactionEmoji: \"thumbsup\"}, before: $before, after: $after, first: $limit) {",
        "    workItems(types: EPIC, not: {myReactionEmoji: \"thumbsup\"}, before: $before, after: $after, first: $limit, includeDescendants: true, excludeProjects: true) {",
        "    workItems(not: {myReactionEmoji: \"thumbsup\"}, before: $before, after: $after, first: $limit) {"
    ),

    test_my_reaction_invalid: (
        "myreaction = ~\"invalid\"",
        "Error: `myreaction` cannot be compared with `~invalid`. Supported value types: `String`.",
        "Error: `myreaction` cannot be compared with `~invalid`. Supported value types: `String`.",
        "Error: `myreaction` cannot be compared with `~invalid`. Supported value types: `String`."
    ),
}

#[test]
fn test_fields_in_output() {
    let query = "project = \"gitlab-org/gitlab\" and type = Issue";
    let context = json!({
        "fields": "title, updated, author, assignees, labels, labels('After*'), lastComment",
    });

    let output = serde_json::from_str::<JsonValue>(&compile(query, &context.to_string())).unwrap();

    assert_eq!(
        output["fields"],
        json!([
            { "key": "title", "label": "Title", "name": "title" },
            { "key": "updated", "label": "Updated", "name": "updatedAt" },
            { "key": "author", "label": "Author", "name": "author" },
            { "key": "assignees", "label": "Assignees", "name": "assignees" },
            { "key": "labels", "label": "Labels", "name": "labels" },
            { "key": "labels_After*", "label": "Labels: After", "name": "labels" },
            { "key": "lastComment", "label": "Last comment", "name": "lastComment" }
        ])
    );
}

#[test]
fn test_fields_in_output_with_aggregate() {
    let query = "project = \"gitlab-org/gitlab\" and type = MergeRequest and merged >= 2025-01-01 and merged <= 2025-01-02";
    let context = json!({
        "fields": "id, title",
        "aggregate": {
            "dimensions": "timeSegment(2d) on merged as 'Date merged'",
            "metrics": "count as 'Total MRs', totalTimeToMerge as 'TTTM'",
        },
    });

    let output = serde_json::from_str::<JsonValue>(&compile(query, &context.to_string())).unwrap();
    assert_eq!(
        output["fields"],
        json!([
            { "key": "merged", "label": "Date merged", "name": "mergedAt" },
            { "key": "count", "label": "Total MRs", "name": "count" },
            { "key": "totalTimeToMerge", "label": "TTTM", "name": "totalTimeToMerge" }
        ])
    );
}
